/*
 * Toolkit GUI, an application built for operating pinkRF's signal generators.
 *
 * Contact: https://www.pinkrf.com/contact/
 * Copyright © 2018-2024 pinkRF B.V
 * GNU General Public License version 3.
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License along with this program. If not, see https://www.gnu.org/licenses/
 *
 * Author: Iordan Svechtarov
 */

#include "gui_1channel_odc.h"
#include "ui_gui_1channel_odc.h"
#include "miscellaneous.h"
#include "serial_v2.h"
#include <iostream>
#include <math.h>
#include <QCoreApplication>
#include <QCursor>
#include <QDebug>
#include <QDir>
#include <QMessageBox>
#include <QNetworkInterface>
#include <QtSerialPort/QSerialPort>
#include <QTimer>
#include <QThread>

QT_USE_NAMESPACE

GUI_1channel_ODC::GUI_1channel_ODC(QWidget *parent) : QMainWindow(parent),
#ifdef ETHERCAT
	ethercatModule(*this),
#endif
ui(new Ui::GUI_1channel_ODC)
{
	ui->setupUi(this);

	firmware_version_requirement[0] = 1;
	firmware_version_requirement[1] = 9;
	firmware_version_requirement[2] = 35;
	firmware_version_requirement[3] = 0;

	GUI_version = "1.1.4";
	qDebug() << "GUI version: " << GUI_version;
	ui->GUI_version_label_2->setText(GUI_version);

	/* get the names of the serialports, as well as all the other startup values from config.txt */
	config = new ConfigHandler(QCoreApplication::applicationDirPath() + "/config.txt");

	//Setup Data Logger
	QStringList loggercolumns = {"RF_enable", "Frequency", "Power_Setpoint_watt", "Power_FWD_watt", "Power_RFL_watt", "S11_Reflection", "PWM_Frequency", "PWM_Duty_Cycle", "PA_Temperature", "SGx_Error"};
	for (int i = 1; i < (config->get_PSU_count() + 1); i++)
	{
//		loggercolumns.append("PSU_Enable_" + QString::number(i));
		loggercolumns.append("PSU_Voltage_" + QString::number(i));
		loggercolumns.append("PSU_Current_" + QString::number(i));
		loggercolumns.append("PSU_Power_" + QString::number(i));
	}
	loggercolumns.append("PSU_Power_Efficiency");
	loggercolumns.append("PSU_Power_Dissipation");
	datalogger = new LoggingClass(config->get_logging_directory(), "log", loggercolumns);

	/*******************************************************************
	 * Setup channels
	 *******************************************************************/
	RF_system_constructor = new RF_System_1channel_ODC();

	//Connect Modbus enable
	connect(this, &GUI_1channel_ODC::signal_set_modbus_enable, RF_system_constructor->mbServer, &ModbusServer::set_listen_enable);
	connect(RF_system_constructor->mbServer, &ModbusServer::signal_modbus_connected_enable, this, &GUI_1channel_ODC::handler_modbus_connected_enable);
	
	//Connect EtherCAT enable
	#ifdef ETHERCAT
		connect(this, &GUI_1channel_ODC::signal_set_ethercat_enable, &ethercatModule, &EtherCATModule::setEnabled);
	#endif
	
	if (config->get_ok_signal_pin() > 0)
	{
		handler_interlock_enable_get(false);	//Set the indicator in a defined state before connecting the slot and signal.
		connect(RF_system_constructor, &RF_System_1channel_ODC::signal_interlock_enable, this, &GUI_1channel_ODC::handler_interlock_enable_get);
	}

	/* Activate Channels */
	for (int i = 0; i < RF_System::Channels->count(); i++)
	{
		connect(this, &GUI_1channel_ODC::signal_channelInit, RF_System::Channels->at(i), &RF_Channel::Initialize);
		connect(this, &GUI_1channel_ODC::signal_timerStart, RF_System::Channels->at(i), &RF_Channel::Timer_Readings_Start);
		connect(this, &GUI_1channel_ODC::signal_timerStop, RF_System::Channels->at(i), &RF_Channel::Timer_Readings_Stop);
		connect(this, &GUI_1channel_ODC::signal_execute_reset_SGx, RF_System::Channels->at(i), &RF_Channel::Execute_Reset_SGx);
		connect(this, &GUI_1channel_ODC::signal_execute_reset_protection, RF_System::Channels->at(i), &RF_Channel::Execute_Reset_Protection);
		connect(this, &GUI_1channel_ODC::signal_execute_error_clear, RF_System::Channels->at(i), &RF_Channel::Execute_Error_Clear);
		connect(this, &GUI_1channel_ODC::signal_set_error_clearing_enable, RF_System::Channels->at(i), &RF_Channel::Set_Error_Clearing_Enable);

		connect(RF_System::Channels->at(i), &RF_Channel::signal_firmware_version_get, this, &GUI_1channel_ODC::handler_firmware_version_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_error_get, this, &GUI_1channel_ODC::handler_error_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_reset_detected, this, &GUI_1channel_ODC::handler_reset_detected);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_temperature_PA_get, this, &GUI_1channel_ODC::handler_temperature_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_power_control_mode_get, this, &GUI_1channel_ODC::handler_power_control_mode_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_clock_source_get, this, &GUI_1channel_ODC::handler_clock_source_get);

		connect(RF_System::Channels->at(i), &RF_Channel::signal_RF_enable_get, this, &GUI_1channel_ODC::handler_RF_enable_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_PA_power_readings, this, &GUI_1channel_ODC::handler_PA_power_readings);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_power_get, this, &GUI_1channel_ODC::handler_power_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_VGA_attenuation_get, this, &GUI_1channel_ODC::handler_VGA_attenuation_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_IQMod_magnitude_get, this, &GUI_1channel_ODC::handler_IQMod_magnitude_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_frequency_get, this, &GUI_1channel_ODC::handler_frequency_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_phase_get, this, &GUI_1channel_ODC::handler_phase_get);

		connect(RF_System::Channels->at(i), &RF_Channel::signal_CW_enable_get, this, &GUI_1channel_ODC::handler_CW_enable_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_DLL_enable_get, this, &GUI_1channel_ODC::handler_DLL_enable_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_DLL_settings_get, this, &GUI_1channel_ODC::handler_DLL_settings_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_ALL_compatible_get, this, &GUI_1channel_ODC::handler_ALL_compatible_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_ALL_enable_get, this, &GUI_1channel_ODC::handler_ALL_enable_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_ALL_settings_get, this, &GUI_1channel_ODC::handler_ALL_settings_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_DVGA_compatible_get, this, &GUI_1channel_ODC::handler_DVGA_compatible_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_DVGA_forward_settings_get, this, &GUI_1channel_ODC::handler_DVGA_forward_settings_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_DVGA_reflected_settings_get, this, &GUI_1channel_ODC::handler_DVGA_reflected_settings_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_PWM_settings_get, this, &GUI_1channel_ODC::handler_PWM_settings_get);		//Also covers External Triggering.
		connect(RF_System::Channels->at(i), &RF_Channel::signal_SWP_settings_get, this, &GUI_1channel_ODC::handler_SWP_settings_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_SWP_measurements_get, this, &GUI_1channel_ODC::handler_SWP_measurement_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_PID_settings_get, this, &GUI_1channel_ODC::handler_PID_settings_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_PSU_IU_reading_get, this, &GUI_1channel_ODC::handler_PSU_IU_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_PSU_power_efficiency_get, this, &GUI_1channel_ODC::handler_PSU_power_efficiency_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_PSU_enable_combined_get, this, &GUI_1channel_ODC::handler_PSU_enable_combined_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_PSU_voltage_setpoint_get, this, &GUI_1channel_ODC::handler_PSU_voltage_setpoint_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_PSU_current_limit_get, this, &GUI_1channel_ODC::handler_PSU_current_limit_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_PSU_dissipation_get, this, &GUI_1channel_ODC::handler_PSU_dissipation_get);

		connect(this, &GUI_1channel_ODC::signal_request_PA_power_update, RF_System::Channels->at(i), &RF_Channel::Update_PA_Power);

		connect(this, &GUI_1channel_ODC::signal_RF_enable, RF_System::Channels->at(i), &RF_Channel::Set_RF_enable);
		connect(this, &GUI_1channel_ODC::signal_setClock_source, RF_System::Channels->at(i), &RF_Channel::Set_Clock_Source);
		connect(this, &GUI_1channel_ODC::signal_set_power_control_mode, RF_System::Channels->at(i), &RF_Channel::Set_Power_Control_Mode);
		connect(this, &GUI_1channel_ODC::signal_setExternal_triggering_enable, RF_System::Channels->at(i), &RF_Channel::Set_External_Triggering_Enable_Configuration);

		connect(this, &GUI_1channel_ODC::signal_setVGA_attenuation, RF_System::Channels->at(i), &RF_Channel::Set_VGA_attenuation);
		connect(this, &GUI_1channel_ODC::signal_setIQMod_magnitude, RF_System::Channels->at(i), &RF_Channel::Set_IQMod_Magnitude);
		connect(this, &GUI_1channel_ODC::signal_setPowerWatt, RF_System::Channels->at(i), &RF_Channel::Set_Power_Watt);
		connect(this, &GUI_1channel_ODC::signal_setPowerdBm, RF_System::Channels->at(i), &RF_Channel::Set_Power_dBm);
		connect(this, &GUI_1channel_ODC::signal_setFrequency, RF_System::Channels->at(i), &RF_Channel::Set_Frequency);
		connect(this, &GUI_1channel_ODC::signal_setPhase, RF_System::Channels->at(i), &RF_Channel::Set_Phase);

//		connect(this, &GUI_1channel_ODC::signal_setCW_enable, RF_System::Channels->at(i), &RF_Channel::Set_CW_enable);
		connect(this, &GUI_1channel_ODC::signal_setPWM_enable, RF_System::Channels->at(i), &RF_Channel::Set_PWM_enable);
		connect(this, &GUI_1channel_ODC::signal_setPWM_frequency, RF_System::Channels->at(i), &RF_Channel::Set_PWM_Frequency);
		connect(this, &GUI_1channel_ODC::signal_setPWM_duty_cycle, RF_System::Channels->at(i), &RF_Channel::Set_PWM_Duty_Cycle);

		connect(this, &GUI_1channel_ODC::signal_setDLL_enable, RF_System::Channels->at(i), &RF_Channel::Set_DLL_enable);
		connect(this, &GUI_1channel_ODC::signal_setDLL_frequency_limit_lower, RF_System::Channels->at(i), &RF_Channel::Set_DLL_Frequency_Lower_Limit);
		connect(this, &GUI_1channel_ODC::signal_setDLL_frequency_limit_upper, RF_System::Channels->at(i), &RF_Channel::Set_DLL_Frequency_Upper_Limit);
		connect(this, &GUI_1channel_ODC::signal_setDLL_start_frequency, RF_System::Channels->at(i), &RF_Channel::Set_DLL_Frequency_Start);
		connect(this, &GUI_1channel_ODC::signal_setDLL_step_frequency, RF_System::Channels->at(i), &RF_Channel::Set_DLL_Frequency_Step);
		connect(this, &GUI_1channel_ODC::signal_setDLL_threshold, RF_System::Channels->at(i), &RF_Channel::Set_DLL_Frequency_Threshold);
		connect(this, &GUI_1channel_ODC::signal_setDLL_main_delay, RF_System::Channels->at(i), &RF_Channel::Set_DLL_Frequency_Main_Delay);

		connect(this, &GUI_1channel_ODC::signal_setALL_enable, RF_System::Channels->at(i), &RF_Channel::Set_ALL_enable);
		connect(this, &GUI_1channel_ODC::signal_setALL_frequency_limit_upper, RF_System::Channels->at(i), &RF_Channel::Set_ALL_Frequency_Upper_Limit);
		connect(this, &GUI_1channel_ODC::signal_setALL_frequency_limit_lower, RF_System::Channels->at(i), &RF_Channel::Set_ALL_Frequency_Lower_Limit);
		connect(this, &GUI_1channel_ODC::signal_setALL_threshold, RF_System::Channels->at(i), &RF_Channel::Set_ALL_Frequency_Threshold);

		connect(this, &GUI_1channel_ODC::signal_setDVGA_amplifier_forward_enable, RF_System::Channels->at(i), &RF_Channel::Set_DVGA_Amplifier_Forward_Enable);
		connect(this, &GUI_1channel_ODC::signal_setDVGA_amplifier_reflected_enable, RF_System::Channels->at(i), &RF_Channel::Set_DVGA_Amplifier_Reflected_Enable);
		connect(this, &GUI_1channel_ODC::signal_setDVGA_attentuation_forward, RF_System::Channels->at(i), &RF_Channel::Set_DVGA_Attenuation_Forward);
		connect(this, &GUI_1channel_ODC::signal_setDVGA_attentuation_reflected, RF_System::Channels->at(i), &RF_Channel::Set_DVGA_Attenuation_Reflected);

		connect(this, &GUI_1channel_ODC::signal_setPID_kp, RF_System::Channels->at(i), &RF_Channel::Set_PID_Kp);
		connect(this, &GUI_1channel_ODC::signal_setPID_ki, RF_System::Channels->at(i), &RF_Channel::Set_PID_Ki);
		connect(this, &GUI_1channel_ODC::signal_setPID_kd, RF_System::Channels->at(i), &RF_Channel::Set_PID_Kd);
		connect(this, &GUI_1channel_ODC::signal_setPID_setpoint, RF_System::Channels->at(i), &RF_Channel::Set_PID_Setpoint);
		connect(this, &GUI_1channel_ODC::signal_setPID_scaling, RF_System::Channels->at(i), &RF_Channel::Set_PID_Scaling);
		connect(this, &GUI_1channel_ODC::signal_setPID_offset, RF_System::Channels->at(i), &RF_Channel::Set_PID_Offset);

		connect(this, &GUI_1channel_ODC::signal_setSWP_power_dbm, RF_System::Channels->at(i), &RF_Channel::Set_SWP_Power_dBm);
		connect(this, &GUI_1channel_ODC::signal_setSWP_power_watt, RF_System::Channels->at(i), &RF_Channel::Set_SWP_Power_Watt);
		connect(this, &GUI_1channel_ODC::signal_setSWP_start_frequency, RF_System::Channels->at(i), &RF_Channel::Set_SWP_Frequency_Start);
		connect(this, &GUI_1channel_ODC::signal_setSWP_stop_frequency, RF_System::Channels->at(i), &RF_Channel::Set_SWP_Frequency_Stop);
		connect(this, &GUI_1channel_ODC::signal_setSWP_step_frequency, RF_System::Channels->at(i), &RF_Channel::Set_SWP_Frequency_Step);
		connect(this, &GUI_1channel_ODC::signal_execute_SWP, RF_System::Channels->at(i), &RF_Channel::Execute_SWP_dBm);

        connect(this, &GUI_1channel_ODC::signal_setPSU_enable, RF_System::Channels->at(i), &RF_Channel::Set_PSU_Enable);
        connect(this, &GUI_1channel_ODC::signal_setPSU_voltage, RF_System::Channels->at(i), &RF_Channel::Set_PSU_Voltage_Setpoint);
		connect(this, &GUI_1channel_ODC::signal_setPSU_current_limit, RF_System::Channels->at(i), &RF_Channel::Set_PSU_Current_Limit);

		connect(RF_System::Channels->at(i), &RF_Channel::signal_channel_working, this, &GUI_1channel_ODC::handler_channel_working);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_channel_init_failed, this, &GUI_1channel_ODC::handler_channel_init_failed);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_notification, this, &GUI_1channel_ODC::handler_notification_get);

		connect(datalogger, &LoggingClass::signal_datalogging_enable, this, &GUI_1channel_ODC::handler_datalogging_enable_get);
		connect(datalogger, &LoggingClass::signal_datalogging_storage_sufficient, this, &GUI_1channel_ODC::handler_datalogging_storage_sufficient);
		connect(datalogger, &LoggingClass::signal_datalogging_log_deleted, this, &GUI_1channel_ODC::handler_datalogging_log_deleted);

		//Modbus interactions
		connect(RF_system_constructor->mbServer, &ModbusServer::signal_set_datalogging_enable, this, &GUI_1channel_ODC::handler_datalogging);
		connect(datalogger, &LoggingClass::signal_datalogging_enable, RF_system_constructor->mbServer, &ModbusServer::handler_datalogging_enable_get);
		connect(RF_system_constructor->mbServer, &ModbusServer::signal_execute_restart_program, this, &GUI_1channel_ODC::handler_execute_restart_program);
		connect(datalogger, &LoggingClass::signal_datalogging_storage_sufficient, RF_system_constructor->mbServer, &ModbusServer::handler_datalogging_storage_sufficient);

		emit signal_channelInit(i+1);
		emit signal_timerStart(i+1, config->get_polling_rate_data());
	}

	#if defined(Q_OS_LINUX)
    #if defined(I2C_TEMP_SENSOR)
    connect(this, &GUI_1channel_ODC::signal_temperature_sensor_init, RF_system_constructor->sensor, &i2cSensor::Initialize);
	connect(RF_system_constructor->sensor, &i2cSensor::signal_temperature_sensor_get, this, &GUI_1channel_ODC::handler_sensor_temperature_get);
    emit signal_temperature_sensor_init();
	#endif
    #endif

	/* set up button groups, make it so only one button can be checked at a time */
	menu_buttonGroup.addButton(ui->menu_home_Button);
	menu_buttonGroup.addButton(ui->menu_ALL_Button);
	menu_buttonGroup.addButton(ui->menu_DLL_Button);
	menu_buttonGroup.addButton(ui->menu_sweep_Button);
	menu_buttonGroup.addButton(ui->menu_settings_Button);
	menu_buttonGroup.addButton(ui->menu_help_Button);
	menu_buttonGroup.addButton(ui->menu_power_Button);
	menu_buttonGroup.addButton(ui->menu_PID_Button);
	menu_buttonGroup.addButton(ui->menu_PSU_Button);
	menu_buttonGroup.setExclusive(true);

	clock_source_buttonGroup.addButton(ui->pushButton_clock_source_standalone_1);
	clock_source_buttonGroup.addButton(ui->pushButton_clock_source_master_1);
	clock_source_buttonGroup.addButton(ui->pushButton_clock_source_slave_end_1);
	clock_source_buttonGroup.addButton(ui->pushButton_clock_source_slave_inline_1);
	clock_source_buttonGroup.setExclusive(true);

	RCM_buttonGroup.addButton(ui->pushButton_remote_command_OFF_1);
	RCM_buttonGroup.addButton(ui->pushButton_remote_command_USB_1);
	RCM_buttonGroup.addButton(ui->pushButton_remote_command_TCP_1);
	RCM_buttonGroup.setExclusive(true);

	power_control_buttonGroup.addButton(ui->pushButton_power_control_normal_1);
	power_control_buttonGroup.addButton(ui->pushButton_power_control_analog_1);
	power_control_buttonGroup.addButton(ui->pushButton_power_control_feedforward_1);
	power_control_buttonGroup.addButton(ui->pushButton_power_control_PID_1);
	power_control_buttonGroup.setExclusive(true);

	External_triggering_buttonGroup.addButton(ui->pushButton_external_triggering_OFF_1);
	External_triggering_buttonGroup.addButton(ui->pushButton_external_triggering_ON_1);
	External_triggering_buttonGroup.setExclusive(true);

	modbus_buttonGroup.addButton(ui->pushButton_modbus_OFF_1);
	modbus_buttonGroup.addButton(ui->pushButton_modbus_TCP_slave_1);
	modbus_buttonGroup.setExclusive(true);
	
	ethercat_buttonGroup.addButton(ui->pushButton_etherCAT_OFF_1);
	ethercat_buttonGroup.addButton(ui->pushButton_etherCAT_ON_1);
	ethercat_buttonGroup.setExclusive(true);

	logging_buttonGroup.addButton(ui->pushButton_data_logging_OFF_1);
	logging_buttonGroup.addButton(ui->pushButton_data_logging_ON_1);
	logging_buttonGroup.setExclusive(true);

	/* set maximum rows on the RCM text edit. */
	ui->settings_plainTextEdit->setMaximumBlockCount(100);

	/* set Watt default */
	set_power_reading_units_watt(true);	//true = watt, false = dBm

	/* set Home menu default */
	ui->menu_home_Button->click();		//Always start from the Home menu.
	ui->stackedWidget_5->setCurrentIndex(ui->stackedWidget_5->indexOf(ui->page_overview));

	/* hide the DLL frequency indicator by default */
	ui->label_DLL_frequency_lock_1->setVisible(false);
	ui->label_DLL_frequency_lock_2->setVisible(false);
	ui->label_DLL_frequency_lock_3->setVisible(false);

	/* Hide the notifications by default */
	ui->label_notification->setVisible(false);
	ui->label_error_message->setVisible(false);
	ui->label_low_storage_1->setVisible(false);

	//
	// TODO:
	// Implement handler_identity_get
	//
	ui->SG_board_identity_label_1->setVisible(false);
	ui->SG_board_identity_label_2->setVisible(false);

    /* Phase control */
	#error Previously used the legacy splitter_mode config value. This needs to be updated. See handler_splitter_configuration from gui_1channel
	ui->phase_label->setEnabled(config->get_support_phase_control() && config->get_splitter_channel_disable() == 0);
	ui->pushButton_phase_1->setEnabled(config->get_support_phase_control() && config->get_splitter_channel_disable() == 0);

	/* Declare the timers. */
	notification_timer = new QTimer(this);		//Used for displaying temporary warning messages.

	//
	// TODO:
	// Get TCP RCM working and Remove these plz.
	//
	/* Initiate the UART connection & prepare for TCP connection. */
	serial_Setup(RCM_port, config->get_RCM_port_USB());	//Port for external USB communication.
	serial_Setup(SG_port, config->get_output_port());	//Port for SGx connection.
	tcpServer = new QTcpServer(this);
	tcpSocket = new QTcpSocket(this);

	/* show or hide the various mode controls in settings menu */
	coherency_supported = (config->get_support_coherency_standalone() ||
						   config->get_support_coherency_LVDS_master() ||
						   config->get_support_coherency_LVDS_slave_end() ||
						   config->get_support_coherency_LVDS_slave_inline() ||
						   config->get_support_coherency_AUX_master() ||
						   config->get_support_coherency_AUX_slave_end());

	ui->pushButton_clock_source_standalone_1->setVisible(config->get_support_coherency_standalone());
	ui->pushButton_clock_source_standalone_1->setEnabled(config->get_support_coherency_standalone());
	ui->pushButton_clock_source_master_1->setVisible(config->get_support_coherency_LVDS_master());
	ui->pushButton_clock_source_master_1->setEnabled(config->get_support_coherency_LVDS_master());
	ui->pushButton_clock_source_slave_end_1->setVisible(config->get_support_coherency_LVDS_slave_end());
	ui->pushButton_clock_source_slave_end_1->setEnabled(config->get_support_coherency_LVDS_slave_end());
	ui->pushButton_clock_source_slave_inline_1->setVisible(config->get_support_coherency_LVDS_slave_inline());
	ui->pushButton_clock_source_slave_inline_1->setEnabled(config->get_support_coherency_LVDS_slave_inline());
//	ui->pushButton_clock_source_AUX_master_1->setVisible(config->get_support_coherency_AUX_master());
//	ui->pushButton_clock_source_AUX_master_1->setEnabled(config->get_support_coherency_AUX_master());
//	ui->pushButton_clock_source_AUX_slave_end_1->setVisible(config->get_support_coherency_AUX_slave_end());
//	ui->pushButton_clock_source_AUX_slave_end_1->setEnabled(config->get_support_coherency_AUX_slave_end());
	ui->label_clock_source_1->setEnabled(coherency_supported);
	ui->label_clock_source_1->setVisible(coherency_supported);


	RCM_supported = config->get_support_RCM_mode_USB_live();
	ui->RCM_label->setEnabled(RCM_supported);
	ui->RCM_label->setVisible(RCM_supported);
	ui->pushButton_remote_command_OFF_1->setVisible(RCM_supported);
	ui->pushButton_remote_command_OFF_1->setEnabled(RCM_supported);
	ui->pushButton_remote_command_USB_1->setVisible(config->get_support_RCM_mode_USB_live());
	ui->pushButton_remote_command_USB_1->setEnabled(config->get_support_RCM_mode_USB_live());
	//
	// TODO:
	// TCP RCM mode is currently unsupported!
	//
//	ui->pushButton_remote_command_TCP_1->setVisible(config->get_support_RCM_mode());
//	ui->pushButton_remote_command_TCP_1->setEnabled(config->get_support_RCM_mode());
	ui->pushButton_remote_command_TCP_1->setVisible(false);
	ui->pushButton_remote_command_TCP_1->setEnabled(false);

	ui->pushButton_etherCAT_OFF_1->setVisible(config->get_support_ethercat_mode());
	ui->pushButton_etherCAT_ON_1->setVisible(config->get_support_ethercat_mode());
    ui->label_etherCAT_mode_1->setVisible(config->get_support_ethercat_mode());

	ui->pushButton_power_control_normal_1->setVisible(config->get_support_power_control_modes());
	ui->pushButton_power_control_normal_1->setEnabled(config->get_support_power_control_modes());
	ui->pushButton_power_control_analog_1->setVisible(config->get_support_power_control_modes() && config->get_support_AIS_mode());
	ui->pushButton_power_control_analog_1->setEnabled(config->get_support_power_control_modes() && config->get_support_AIS_mode());
	ui->pushButton_power_control_PID_1->setVisible(config->get_support_power_control_modes() && config->get_support_PID_control());
	ui->pushButton_power_control_PID_1->setEnabled(config->get_support_power_control_modes() && config->get_support_PID_control());
	ui->pushButton_power_control_feedforward_1->setVisible(config->get_support_power_control_modes() && config->get_support_feedforward_control());
	ui->pushButton_power_control_feedforward_1->setEnabled(config->get_support_power_control_modes() && config->get_support_feedforward_control());
	ui->label_power_control_mode_1->setEnabled(config->get_support_power_control_modes());
	ui->label_power_control_mode_1->setVisible(config->get_support_power_control_modes());
	
	ui->pushButton_external_triggering_OFF_1->setVisible(config->get_support_external_triggering_mode());
	ui->pushButton_external_triggering_OFF_1->setEnabled(config->get_support_external_triggering_mode());
	ui->pushButton_external_triggering_ON_1->setVisible(config->get_support_external_triggering_mode());
	ui->pushButton_external_triggering_ON_1->setEnabled(config->get_support_external_triggering_mode());
	ui->External_triggering_label->setVisible(config->get_support_external_triggering_mode());
	ui->External_triggering_label->setEnabled(config->get_support_external_triggering_mode());

	ui->pushButton_modbus_OFF_1->setVisible(config->get_support_modbus_mode());
	ui->pushButton_modbus_OFF_1->setEnabled(config->get_support_modbus_mode());
	ui->pushButton_modbus_TCP_slave_1->setVisible(config->get_support_modbus_mode());
	ui->pushButton_modbus_TCP_slave_1->setEnabled(config->get_support_modbus_mode());
	ui->label_modbus_mode_1->setVisible(config->get_support_modbus_mode());
	ui->label_modbus_mode_1->setEnabled(config->get_support_modbus_mode());

	ui->pushButton_data_logging_ON_1->setVisible(config->get_support_logging());
	ui->pushButton_data_logging_ON_1->setEnabled(config->get_support_logging());
	ui->pushButton_data_logging_OFF_1->setVisible(config->get_support_logging());
	ui->pushButton_data_logging_OFF_1->setEnabled(config->get_support_logging());
	ui->label_data_logging_1->setVisible(config->get_support_logging());
	ui->label_data_logging_1->setEnabled(config->get_support_logging());

	ui->menu_home_Button->setVisible(config->get_menu_home_enabled());			//enable or disable the home menu
	ui->menu_home_Button->setEnabled(config->get_menu_home_enabled());
	ui->menu_DLL_Button->setVisible(config->get_menu_DLL_enabled());			//enable or disable the DLL menu
	ui->menu_DLL_Button->setEnabled(config->get_menu_DLL_enabled());
	ui->menu_PID_Button->setVisible(config->get_menu_PID_enabled());			//enable or disable the PID menu
	ui->menu_PID_Button->setEnabled(config->get_menu_PID_enabled());
	ui->menu_sweep_Button->setVisible(config->get_menu_sweep_enabled());		//enable or disable the Sweep menu
	ui->menu_sweep_Button->setEnabled(config->get_menu_sweep_enabled());
	ui->menu_PSU_Button->setVisible(config->get_menu_PSU_enabled());			//enable or disable the PSU menu
	ui->menu_PSU_Button->setEnabled(config->get_menu_PSU_enabled());
	ui->menu_settings_Button->setVisible(config->get_menu_settings_enabled());	//enable or disable the settings menu
	ui->menu_settings_Button->setEnabled(config->get_menu_settings_enabled());
	ui->menu_help_Button->setVisible(config->get_menu_help_enabled());			//enable or disable the help menu
	ui->menu_help_Button->setEnabled(config->get_menu_help_enabled());
	ui->menu_power_Button->setVisible(config->get_menu_power_enabled());		//enable or disable the power menu
	ui->menu_power_Button->setEnabled(config->get_menu_power_enabled());

	ui->pushButton_CW_enable_1->setVisible(config->get_controls_CW_enabled());		//enable or disable CW button
	ui->pushButton_CW_enable_1->setEnabled(config->get_controls_CW_enabled());
	ui->pushButton_PWM_enable_1->setVisible(config->get_controls_PWM_enabled());	//enable or disable PWM button
	ui->pushButton_PWM_enable_1->setEnabled(config->get_controls_PWM_enabled());
	ui->pushButton_DLL_enable_1->setVisible(config->get_controls_DLL_enabled());	//enable or disable DLL button
	ui->pushButton_DLL_enable_1->setEnabled(config->get_controls_DLL_enabled());

	//If 3v3 OK signal is assigned to a pin; PSU's should be turned on by the 24V signal, not by the user.
	if (config->get_ok_signal_pin() > 0)
	{
		ui->pushButton_PSU_enable_1->setEnabled(false);
		ui->pushButton_PSU_voltage_1->setEnabled(false);
	}
	else
	{
		ui->label_interlock_1->setVisible(false);
		ui->label_interlock_2->setVisible(false);
	}

	/* Create Numpad object + connect input confirmation with the respective handler */
	numpad = new Numpad(5,4, ui->numpadLabel_display, ui->numpadLabel_unit, ui->numpadButton_period, ui->numpadButton_sign);
	connect(numpad, &Numpad::value_confirmed_signal, this, &GUI_1channel_ODC::numpad_value_confirmed_handler);

	/* optionally enable the temperature measurements */
	ui->label_temperature_1->setVisible(config->get_read_temperature());
	ui->label_temperature_2->setVisible(config->get_read_temperature());
	ui->label_temperature_3->setVisible(config->get_read_temperature());

	#ifdef I2C_TEMP_SENSOR
	ui->label_temperature_1->setVisible(true);
	ui->label_temperature_2->setVisible(true);
	ui->label_temperature_3->setVisible(true);
	#endif

	/* Load a stylesheet */
	QFile stylesheetfile(QCoreApplication::applicationDirPath() + "/theme.css");
	QString style;
	stylesheetfile.open(QFile::ReadOnly);
	style = stylesheetfile.readAll();
	stylesheetfile.close();

	/* Set stylesheet across the board */
	this->setStyleSheet(style);

	//
	// TODO:
	// I bet this can be handled from the css file?
	//
	/* This is necessary so the logo image loads correctly regardless of the working path. Image needs to be in the same folder as the executable. */
	ui->logo_label->setStyleSheet("image: url(" + QCoreApplication::applicationDirPath() + "/logo.png);\nimage-position: top;\n");

	// Set the start condition for RCM.
	handler_RCM_mode();

	// Set the start condition for Logging
	if (config->get_support_logging())
	{
		handler_datalogging(config->get_logging_enabled());
	}

	// Set the start condition for Modbus
	handler_modbus_mode();
	
	// Set the start condition for EtherCAT
	handler_ethercat_mode();

	//Configure Power Supplies:
	if (config->get_PSU_count() > 0)
	{
		if (config->get_PSU_voltage() > 0)
		{
			emit signal_setPSU_voltage(1, config->get_PSU_voltage());
		}
		if (config->get_PSU_current_limit() > 0)
		{
			emit signal_setPSU_current_limit(1, config->get_PSU_current_limit());
		}
	}

	#ifdef DEBUG
	show_notification("THIS IS A DEBUG BUILD!");
	ui->settings_plainTextEdit->appendPlainText(">\t THIS IS A DEBUG BUILD!");
	#endif
}

GUI_1channel_ODC::~GUI_1channel_ODC()
{
	RF_system_constructor->~RF_System_1channel_ODC();
	delete ui;
}

/**********************************************************************************************************************************************************************************
 * UTILITY FUNCTIONS
 * *******************************************************************************************************************************************************************************/

/* Enable or disable automatic error clearing */
void GUI_1channel_ODC::configure_auto_error_clearing()
{
	//
	// TODO:
	// It's a little bit iffy that button states are used for a check, instead of something more proper like DEFINED values for RCM and Modbus OFF.
	// If these were SG commands we could poll the Get command. Maybe we need something similar internally for consistency and safety's sake?
	//

	//
	// TODO:
	// The reason auto-error clearing wasn't breaking if RCM_support was disabled, but was with modbus, is that RCM_OFF_button is checked by default in the GUI.
	// But the code shouldn't rely on the default state of the UI to perform it's job properly! That's too farfetched...
	//

	bool enable = true;
	if (RCM_supported == true)
	{
		enable &= ui->pushButton_remote_command_OFF_1->isChecked();
	}

	if (config->get_support_modbus_mode())
	{
		enable &= ui->pushButton_modbus_OFF_1->isChecked();
	}
	
	if (config->get_support_ethercat_mode())
	{
		enable &= ui->pushButton_etherCAT_OFF_1->isChecked();
	}

//	bool enable = ui->pushButton_remote_command_OFF_1->isChecked() && ui->pushButton_modbus_OFF_1->isChecked();
	emit signal_set_error_clearing_enable(1, enable);
}

/**********************************************************************************************************************************************************************************
 * Parameter buttons
 * *******************************************************************************************************************************************************************************/

// HOME
void GUI_1channel_ODC::on_pushButton_VGA_attenuation_1_clicked(){
	numpad_value = RF_System::Channels->at(0)->VGA_Attenuation();
	numpad_parameter_select(ui->pushButton_VGA_attenuation_1, &numpad_value, "dB", 1);
}
void GUI_1channel_ODC::on_pushButton_IQMod_magnitude_1_clicked(){
	numpad_value = RF_System::Channels->at(0)->IQMod_Magnitude();
	numpad_parameter_select(ui->pushButton_IQMod_magnitude_1, &numpad_value, "%", 1);
}
void GUI_1channel_ODC::on_pushButton_power_watt_1_clicked(){
	numpad_value = RF_System::Channels->at(0)->Power_watt();
	numpad_parameter_select(ui->pushButton_power_watt_1, &numpad_value, "watt", 1);
}
void GUI_1channel_ODC::on_pushButton_power_dbm_1_clicked(){
	numpad_value = RF_System::Channels->at(0)->Power_dBm();
	numpad_parameter_select(ui->pushButton_power_dbm_1, &numpad_value, "dBm", 1);
}
void GUI_1channel_ODC::on_pushButton_frequency_1_clicked(){
	numpad_value = RF_System::Channels->at(0)->Frequency();
	numpad_parameter_select(ui->pushButton_frequency_1, &numpad_value, "MHz", 3);
}
void GUI_1channel_ODC::on_pushButton_phase_1_clicked(){
	numpad_value = RF_System::Channels->at(0)->Phase();
	numpad_parameter_select(ui->pushButton_phase_1, &numpad_value, "°", 0, 1 , false);
}
void GUI_1channel_ODC::on_pushButton_PWM_frequency_1_clicked(){
	numpad_value = RF_System::Channels->at(0)->PWM_Frequency();
	numpad_parameter_select(ui->pushButton_PWM_frequency_1, &numpad_value, "Hz", 0, 1, false);
}
void GUI_1channel_ODC::on_pushButton_PWM_duty_cycle_1_clicked(){
	numpad_value = RF_System::Channels->at(0)->PWM_Duty_Cycle();
	numpad_parameter_select(ui->pushButton_PWM_duty_cycle_1, &numpad_value, "%", 0, 1, false);
}
void GUI_1channel_ODC::on_pushButton_DLL_step_frequency_1_clicked(){
	numpad_value = RF_System::Channels->at(0)->DLL_Step_Frequency();
	numpad_parameter_select(ui->pushButton_DLL_step_frequency_1, &numpad_value, "MHz", 3);
}
void GUI_1channel_ODC::on_pushButton_DLL_threshold_1_clicked(){
	numpad_value = RF_System::Channels->at(0)->DLL_Threshold();
	numpad_parameter_select(ui->pushButton_DLL_threshold_1, &numpad_value, "dB");
}
void GUI_1channel_ODC::on_pushButton_ALL_threshold_1_clicked(){
	numpad_value = RF_System::Channels->at(0)->ALL_Threshold();
	numpad_parameter_select(ui->pushButton_ALL_threshold_1, &numpad_value, "V");
}


// ALL
void GUI_1channel_ODC::on_pushButton_ALL_frequency_limit_upper_1_clicked(){
	numpad_value = RF_System::Channels->at(0)->ALL_Frequency_Limit_Upper();
	numpad_parameter_select(ui->pushButton_ALL_frequency_limit_upper_1, &numpad_value, "V");
}
void GUI_1channel_ODC::on_pushButton_ALL_frequency_limit_lower_1_clicked(){
	numpad_value = RF_System::Channels->at(0)->ALL_Frequency_Limit_Lower();
	numpad_parameter_select(ui->pushButton_ALL_frequency_limit_lower_1, &numpad_value, "V");
}
void GUI_1channel_ODC::on_pushButton_ALL_threshold_2_clicked(){
	numpad_value = RF_System::Channels->at(0)->ALL_Threshold();
	numpad_parameter_select(ui->pushButton_ALL_threshold_2, &numpad_value, "V");
}
void GUI_1channel_ODC::on_pushButton_DVGA_attenuation_forward_1_clicked(){
	numpad_value = RF_System::Channels->at(0)->DVGA_Attenuation_Forward();
	numpad_parameter_select(ui->pushButton_DVGA_attenuation_forward_1, &numpad_value, "dB");
}
void GUI_1channel_ODC::on_pushButton_DVGA_attenuation_reflected_1_clicked(){
	numpad_value = RF_System::Channels->at(0)->DVGA_Attenuation_Reflected();
	numpad_parameter_select(ui->pushButton_DVGA_attenuation_reflected_1, &numpad_value, "dB");
}

void GUI_1channel_ODC::on_pushButton_ALL_reference_open_clicked()
{
	emit signal_RF_enable(1,false);
	ui->stackedWidget_3->setCurrentIndex(ui->stackedWidget_3->indexOf(ui->ALL_reference_page));
}

void GUI_1channel_ODC::on_pushButton_DVGA_amplifier_forward_1_clicked()
{
	emit signal_setDVGA_amplifier_forward_enable(1, ui->pushButton_DVGA_amplifier_forward_1->isChecked());
}

void GUI_1channel_ODC::on_pushButton_DVGA_amplifier_reflected_1_clicked()
{
	emit signal_setDVGA_amplifier_reflected_enable(1, ui->pushButton_DVGA_amplifier_reflected_1->isChecked());
}

/* The button on the reference page that sends you back to the main */
void GUI_1channel_ODC::on_pushButton_ALL_reference_back_clicked()
{
	ui->stackedWidget_3->setCurrentIndex(ui->stackedWidget_3->indexOf(ui->Main_page));
}

// SWEEP
void GUI_1channel_ODC::on_pushButton_SWP_start_frequency_1_clicked(){
	numpad_value = RF_System::Channels->at(0)->SWP_Start_Frequency();
//	numpad_parameter_select(ui->pushButton_SWP_start_frequency_1, &numpad_value, "MHz", 3, 0.000001);
	numpad_parameter_select(ui->pushButton_SWP_start_frequency_1, &numpad_value, "MHz", 3);
}
void GUI_1channel_ODC::on_pushButton_SWP_stop_frequency_1_clicked(){
	numpad_value = RF_System::Channels->at(0)->SWP_Stop_Frequency();
//	numpad_parameter_select(ui->pushButton_SWP_stop_frequency_1, &numpad_value, "MHz", 3, 0.000001);
	numpad_parameter_select(ui->pushButton_SWP_stop_frequency_1, &numpad_value, "MHz", 3);
}
void GUI_1channel_ODC::on_pushButton_SWP_step_frequency_1_clicked(){
	numpad_value = RF_System::Channels->at(0)->SWP_Step_Frequency();
//	numpad_parameter_select(ui->pushButton_SWP_step_frequency_1, &numpad_value, "MHz", 3, 0.000001);
	numpad_parameter_select(ui->pushButton_SWP_step_frequency_1, &numpad_value, "MHz", 3);
}
void GUI_1channel_ODC::on_pushButton_SWP_power_dbm_1_clicked(){
	numpad_value = RF_System::Channels->at(0)->SWP_Power_dBm();
	numpad_parameter_select(ui->pushButton_SWP_power_dbm_1, &numpad_value, "dBm", 1);
}
void GUI_1channel_ODC::on_pushButton_SWP_power_watt_1_clicked(){
	numpad_value = RF_System::Channels->at(0)->SWP_Power_watt();
	numpad_parameter_select(ui->pushButton_SWP_power_watt_1, &numpad_value, "watt", 1);
}

// DLL
void GUI_1channel_ODC::on_pushButton_DLL_frequency_limit_lower_1_clicked(){
	numpad_value = RF_System::Channels->at(0)->DLL_Frequency_Limit_Lower();
	numpad_parameter_select(ui->pushButton_DLL_frequency_limit_lower_1, &numpad_value, "MHz", 3);
}
void GUI_1channel_ODC::on_pushButton_DLL_frequency_limit_upper_1_clicked(){
	numpad_value = RF_System::Channels->at(0)->DLL_Frequency_Limit_Upper();
	numpad_parameter_select(ui->pushButton_DLL_frequency_limit_upper_1, &numpad_value, "MHz", 3);
}
void GUI_1channel_ODC::on_pushButton_DLL_start_frequency_1_clicked(){
	numpad_value = RF_System::Channels->at(0)->DLL_Start_Frequency();
	numpad_parameter_select(ui->pushButton_DLL_start_frequency_1, &numpad_value, "MHz", 3);
}
void GUI_1channel_ODC::on_pushButton_DLL_step_frequency_2_clicked(){
	numpad_value = RF_System::Channels->at(0)->DLL_Step_Frequency();
	numpad_parameter_select(ui->pushButton_DLL_step_frequency_2, &numpad_value, "MHz", 3);
}
void GUI_1channel_ODC::on_pushButton_DLL_threshold_2_clicked(){
	numpad_value = RF_System::Channels->at(0)->DLL_Threshold();
	numpad_parameter_select(ui->pushButton_DLL_threshold_2, &numpad_value, "dB");
}
void GUI_1channel_ODC::on_pushButton_DLL_main_delay_1_clicked(){
	numpad_value = RF_System::Channels->at(0)->DLL_Main_Delay();
	numpad_parameter_select(ui->pushButton_DLL_main_delay_1, &numpad_value, "ms", 0, 1, false);
}
void GUI_1channel_ODC::on_pushButton_power_dbm_2_clicked(){
	numpad_value = RF_System::Channels->at(0)->Power_dBm();
	numpad_parameter_select(ui->pushButton_power_dbm_2, &numpad_value, "dBm", 1);
}
void GUI_1channel_ODC::on_pushButton_power_watt_2_clicked(){
	numpad_value = RF_System::Channels->at(0)->Power_watt();
	numpad_parameter_select(ui->pushButton_power_watt_2, &numpad_value, "watt", 1);
}

//PID
void GUI_1channel_ODC::on_pushButton_PID_kp_1_clicked(){
	numpad_value = RF_System::Channels->at(0)->PID_Kp();
	numpad_parameter_select(ui->pushButton_PID_kp_1, &numpad_value, "***", 1);
}
void GUI_1channel_ODC::on_pushButton_PID_ki_1_clicked(){
	numpad_value = RF_System::Channels->at(0)->PID_Ki();
	numpad_parameter_select(ui->pushButton_PID_ki_1, &numpad_value, "***", 1);
}
void GUI_1channel_ODC::on_pushButton_PID_kd_1_clicked(){
	numpad_value = RF_System::Channels->at(0)->PID_Kd();
	numpad_parameter_select(ui->pushButton_PID_kd_1, &numpad_value, "***", 1);
}
void GUI_1channel_ODC::on_pushButton_PID_setpoint_1_clicked(){
	numpad_value = RF_System::Channels->at(0)->PID_Setpoint();
	numpad_parameter_select(ui->pushButton_PID_setpoint_1, &numpad_value, "***", 1);
}
void GUI_1channel_ODC::on_pushButton_PID_scaling_1_clicked(){
	numpad_value = RF_System::Channels->at(0)->PID_Scaling();
	numpad_parameter_select(ui->pushButton_PID_scaling_1, &numpad_value, "X", 1);
}
void GUI_1channel_ODC::on_pushButton_PID_offset_1_clicked(){
	numpad_value = RF_System::Channels->at(0)->PID_Offset();
	numpad_parameter_select(ui->pushButton_PID_offset_1, &numpad_value, "+", 1);
//	numpad_parameter_select(ui->pushButton_PID_offset_1, &numpad_value, "±", 1);
}

//PSU
void GUI_1channel_ODC::on_pushButton_PSU_voltage_1_clicked()
{
	numpad_value = RF_System::Channels->at(0)->PSU_Voltage_Setpoint();
	numpad_parameter_select(ui->pushButton_PSU_voltage_1, &numpad_value, "V", 1, 1, false);
}

void GUI_1channel_ODC::on_pushButton_PSU_current_limit_1_clicked()
{
	numpad_value = RF_System::Channels->at(0)->PSU_Current_Limit();
	numpad_parameter_select(ui->pushButton_PSU_current_limit_1, &numpad_value, "A", 1, 1, false);
}

void GUI_1channel_ODC::on_pushButton_PSU_enable_1_clicked(bool checked)
{
	emit signal_setPSU_enable(1, checked);
}



/**********************************************************************************************************************************************************************************
 * NUMPAD BUTTONS
 * *******************************************************************************************************************************************************************************/

void GUI_1channel_ODC::on_numpadButton_0_clicked(){
	numpad->press_0();
}
void GUI_1channel_ODC::on_numpadButton_1_clicked(){
	numpad->press_1();
}
void GUI_1channel_ODC::on_numpadButton_2_clicked(){
	numpad->press_2();
}
void GUI_1channel_ODC::on_numpadButton_3_clicked(){
	numpad->press_3();
}
void GUI_1channel_ODC::on_numpadButton_4_clicked(){
	numpad->press_4();
}
void GUI_1channel_ODC::on_numpadButton_5_clicked(){
	numpad->press_5();
}
void GUI_1channel_ODC::on_numpadButton_6_clicked(){
	numpad->press_6();
}
void GUI_1channel_ODC::on_numpadButton_7_clicked(){
	numpad->press_7();
}
void GUI_1channel_ODC::on_numpadButton_8_clicked(){
	numpad->press_8();
}
void GUI_1channel_ODC::on_numpadButton_9_clicked(){
	numpad->press_9();
}
void GUI_1channel_ODC::on_numpadButton_period_clicked(){
	numpad->press_period();
}
void GUI_1channel_ODC::on_numpadButton_backspace_clicked(){
	numpad->press_delete();
}
void GUI_1channel_ODC::on_numpadButton_clear_all_clicked(){
	numpad->press_delete_all();
}
void GUI_1channel_ODC::on_numpadButton_arrow_left_clicked(){
	numpad->press_indicator_left();
}
void GUI_1channel_ODC::on_numpadButton_arrow_right_clicked(){
	numpad->press_indicator_right();
}
void GUI_1channel_ODC::on_numpadButton_plus_clicked(){
	numpad->press_plus();
}
void GUI_1channel_ODC::on_numpadButton_minus_clicked(){
	numpad->press_minus();
}
void GUI_1channel_ODC::on_numpadButton_ok_clicked(){
	numpad->press_confirm_input();
	ui->stackedWidget_2->setCurrentIndex(ui->stackedWidget_2->indexOf(ui->page_logo));
}

/**********************************************************************************************************************************************************************************
 * FUNCTIONS FOR NUMPAD
 * *******************************************************************************************************************************************************************************/
void GUI_1channel_ODC::numpad_parameter_select(QPushButton *button, double *parameter, QString unit, int precision, double scale,  bool period_enable, bool sign_enable)
{
	numpad->setOutputButton(button);
	numpad->load_value(parameter, unit, period_enable, sign_enable, scale);
	ui->stackedWidget_2->setCurrentIndex(ui->stackedWidget_2->indexOf(ui->page_numpad));
	scale_multiplier = scale;
	this->precision = precision;
	//
	// TODO:
	// is something going wrong here? It seems as if the precisions value is getting ignored (see frequency set)
	//
}

void GUI_1channel_ODC::numpad_value_confirmed_handler(QPushButton *button, double *num)
{
//	ok_check_minmax(button);
	QString str = zeroChopper(QString::number(*num * scale_multiplier,'f', precision));
	button->setText(str);

	//Home Page parameters
	if (button == ui->pushButton_frequency_1){
		emit signal_setFrequency(1, *num);
	}
	else if (button == ui->pushButton_power_dbm_1){
		emit signal_setPowerdBm(1, *num);
	}
	else if (button == ui->pushButton_power_watt_1){
		emit signal_setPowerWatt(1, *num);
	}
	else if (button == ui->pushButton_VGA_attenuation_1){
		emit signal_setVGA_attenuation(1, *num);
	}
	else if (button == ui->pushButton_IQMod_magnitude_1){
		emit signal_setIQMod_magnitude(1, *num);
	}
	else if (button == ui->pushButton_phase_1){
		emit signal_setPhase(1, *num);
	}
	else if (button == ui->pushButton_PWM_frequency_1){
		emit signal_setPWM_frequency(1,*num);
	}
	else if (button == ui->pushButton_PWM_duty_cycle_1){
		emit signal_setPWM_duty_cycle(1,*num);
	}
	else if (button == ui->pushButton_DLL_step_frequency_1){
		emit signal_setDLL_step_frequency(1,*num);
	}
	else if (button == ui->pushButton_DLL_threshold_1){
		emit signal_setDLL_threshold(1,*num);
	}
	else if (button == ui->pushButton_ALL_threshold_1){
		emit signal_setALL_threshold(1,*num);
	}
	//ALL Page parameters
	else if (button == ui->pushButton_ALL_frequency_limit_upper_1){
		emit signal_setALL_frequency_limit_upper(1,*num);
	}
	else if (button == ui->pushButton_ALL_frequency_limit_lower_1){
		emit signal_setALL_frequency_limit_lower(1,*num);
	}
	else if (button == ui->pushButton_ALL_threshold_2){
		emit signal_setALL_threshold(1,*num);
	}
	else if (button == ui->pushButton_DVGA_attenuation_forward_1){
		emit signal_setDVGA_attentuation_forward(1,*num);
	}
	else if (button == ui->pushButton_DVGA_attenuation_reflected_1){
		emit signal_setDVGA_attentuation_reflected(1,*num);
	}

	//DLL Page parameters
	else if (button == ui->pushButton_DLL_frequency_limit_upper_1){
		emit signal_setDLL_frequency_limit_upper(1,*num);
	}
	else if (button == ui->pushButton_DLL_frequency_limit_lower_1){
		emit signal_setDLL_frequency_limit_lower(1,*num);
	}
	else if (button == ui->pushButton_DLL_start_frequency_1){
		emit signal_setDLL_start_frequency(1,*num);
	}
	else if (button == ui->pushButton_DLL_step_frequency_2){
		emit signal_setDLL_step_frequency(1,*num);
	}
	else if (button == ui->pushButton_DLL_threshold_2){
		emit signal_setDLL_threshold(1,*num);
	}
	else if (button == ui->pushButton_DLL_main_delay_1){
		emit signal_setDLL_main_delay(1,*num);
	}
	else if (button == ui->pushButton_power_dbm_2){
		emit signal_setPowerdBm(1, *num);
	}
	else if (button == ui->pushButton_power_watt_2){
		emit signal_setPowerWatt(1, *num);
	}

	//Sweep Page parameters
	else if (button == ui->pushButton_SWP_start_frequency_1){
		emit signal_setSWP_start_frequency(1,*num);
	}
	else if (button == ui->pushButton_SWP_stop_frequency_1){
		emit signal_setSWP_stop_frequency(1,*num);
	}
	else if (button == ui->pushButton_SWP_step_frequency_1){
		emit signal_setSWP_step_frequency(1,*num);
	}
	else if (button == ui->pushButton_SWP_power_dbm_1){
		emit signal_setSWP_power_dbm(1,*num);
	}
	else if (button == ui->pushButton_SWP_power_watt_1){
		emit signal_setSWP_power_watt(1,*num);
	}

	//PID Page parameters
	else if (button == ui->pushButton_PID_kp_1)
	{
		emit signal_setPID_kp(1,*num);
	}
	else if (button == ui->pushButton_PID_ki_1)
	{
		emit signal_setPID_ki(1,*num);
	}
	else if (button == ui->pushButton_PID_kd_1)
	{
		emit signal_setPID_kd(1,*num);
	}
	else if (button == ui->pushButton_PID_setpoint_1)
	{
		emit signal_setPID_setpoint(1,*num);
	}
	else if (button == ui->pushButton_PID_scaling_1)
	{
		emit signal_setPID_scaling(1,*num);
	}
	else if (button == ui->pushButton_PID_offset_1)
	{
		emit signal_setPID_offset(1,*num);
	}

	//PSU Page parameters
	else if (button == ui->pushButton_PSU_voltage_1)
	{
		emit signal_setPSU_voltage(1,*num);
	}
	else if (button == ui->pushButton_PSU_current_limit_1)
	{
		emit signal_setPSU_current_limit(1,*num);
	}

}


/**********************************************************************************************************************************************************************************
 * MENU BUTTONS
 * *******************************************************************************************************************************************************************************/
void GUI_1channel_ODC::on_menu_home_Button_clicked()
{
	menuButton_manager(1);
}
void GUI_1channel_ODC::on_menu_ALL_Button_clicked()
{
	menuButton_manager(2);
}
void GUI_1channel_ODC::on_menu_settings_Button_clicked()
{
	menuButton_manager(3);
}
void GUI_1channel_ODC::on_menu_sweep_Button_clicked()
{
	menuButton_manager(4);
}
void GUI_1channel_ODC::on_menu_help_Button_clicked()
{
	menuButton_manager(5);
}
void GUI_1channel_ODC::on_menu_DLL_Button_clicked()
{
	menuButton_manager(6);
}
void GUI_1channel_ODC::on_menu_PID_Button_clicked()
{
	menuButton_manager(7);
}
void GUI_1channel_ODC::on_menu_PSU_Button_clicked()
{
	menuButton_manager(8);
}

void GUI_1channel_ODC::menuButton_manager(int menuButton_ID)
{
	if(menuButton_ID != 4){ //enable only if the page sweep
		ui->pushButton_SWP_execute_1->setVisible(false);
		ui->pinkline_2->setVisible(false);
	}
	else{
		ui->pushButton_SWP_execute_1->setVisible(true);
		ui->pinkline_2->setVisible(true);
	}
	switch(menuButton_ID){

		//HOME page
		case 1:
			ui->stackedWidget->setCurrentIndex(ui->stackedWidget->indexOf(ui->home_controls_page));
			ui->stackedWidget_2->setCurrentIndex(ui->stackedWidget_2->indexOf(ui->page_logo));
			ui->stackedWidget_3->setCurrentIndex(ui->stackedWidget_3->indexOf(ui->Main_page));
		break;

		//ALL page
		case 2:
			ui->stackedWidget->setCurrentIndex(ui->stackedWidget->indexOf(ui->ALL_controls_page));
			ui->stackedWidget_2->setCurrentIndex(ui->stackedWidget_2->indexOf(ui->page_logo));
			ui->stackedWidget_3->setCurrentIndex(ui->stackedWidget_3->indexOf(ui->Main_page));
		break;

		//SETTINGS page
		case 3:
			ui->stackedWidget_3->setCurrentIndex(ui->stackedWidget_3->indexOf(ui->Settings_page));
			emit signal_RF_enable(1,false);			//disable RF power because user loses sight of the active data.

		break;

		//SWEEP page
		case 4:
			if(ui->pushButton_SWP_execute_1->isChecked() == true)
			{
				ui->stackedWidget_3->setCurrentIndex(ui->stackedWidget_3->indexOf(ui->Sweep_graph_page));
				emit signal_RF_enable(1,false);		//disable RF power because user loses sight of the active data.
			}
			else{
				ui->stackedWidget->setCurrentIndex(ui->stackedWidget->indexOf(ui->sweep_controls_page));
				ui->stackedWidget_2->setCurrentIndex(ui->stackedWidget_2->indexOf(ui->page_logo));
				ui->stackedWidget_3->setCurrentIndex(ui->stackedWidget_3->indexOf(ui->Main_page));
			}
		break;

		//HELP page
		case 5:
			ui->stackedWidget_3->setCurrentIndex(ui->stackedWidget_3->indexOf(ui->Help_page));
		break;

		//DLL page
		case 6:
			ui->stackedWidget->setCurrentIndex(ui->stackedWidget->indexOf(ui->DLL_controls_page));
			ui->stackedWidget_2->setCurrentIndex(ui->stackedWidget_2->indexOf(ui->page_logo));
			ui->stackedWidget_3->setCurrentIndex(ui->stackedWidget_3->indexOf(ui->Main_page));
		break;

		//PID page
		case 7:
			ui->stackedWidget->setCurrentIndex(ui->stackedWidget->indexOf(ui->PID_controls_page));
			ui->stackedWidget_2->setCurrentIndex(ui->stackedWidget_2->indexOf(ui->page_logo));
			ui->stackedWidget_3->setCurrentIndex(ui->stackedWidget_3->indexOf(ui->Main_page));
		break;

		//PSU page
		case 8:
			ui->stackedWidget->setCurrentIndex(ui->stackedWidget->indexOf(ui->PSU_controls_page));
			ui->stackedWidget_2->setCurrentIndex(ui->stackedWidget_2->indexOf(ui->page_logo));
			ui->stackedWidget_3->setCurrentIndex(ui->stackedWidget_3->indexOf(ui->Main_page));
		break;

	default:
		break;
	}
}

/**********************************************************************************************************************************************************************************
 * UTILITY BUTTONS
 * *******************************************************************************************************************************************************************************/
void GUI_1channel_ODC::on_pushButton_RF_enable_1_clicked()
{
	if (config->get_ODC_workaround_enabled())
	{
		on_pushButton_power_control_normal_1_clicked();
	}

	emit signal_RF_enable(1, ui->pushButton_RF_enable_1->isChecked());
}

void GUI_1channel_ODC::on_pushButton_unit_power_reflected_1_clicked(bool checked)
{
	set_power_reading_units_watt(checked);
}

void GUI_1channel_ODC::on_pushButton_unit_power_forward_1_clicked(bool checked)
{
	set_power_reading_units_watt(checked);
}

void GUI_1channel_ODC::on_pushButton_unit_S11_1_clicked(bool checked)
{
	set_power_reading_units_watt(checked);
}

void GUI_1channel_ODC::set_power_reading_units_watt(bool enable)
{
	if(enable == false)
	{
		ui->pushButton_unit_power_reflected_1->setText("dBm");
		ui->pushButton_unit_power_forward_1->setText("dBm");
		ui->pushButton_unit_S11_1->setText("dB");
		ui->S11_title_label->setText("S11:");
		ui->pushButton_unit_S11_1->setChecked(false);
		ui->pushButton_unit_power_forward_1->setChecked(false);
		ui->pushButton_unit_power_reflected_1->setChecked(false);
		ui->pushButton_SWP_unit->setText("Unit: log");
		ui->pushButton_SWP_unit->setChecked(false);
	}
	else
	{
		ui->pushButton_unit_power_reflected_1->setText("watt");
		ui->pushButton_unit_power_forward_1->setText("watt");
		ui->pushButton_unit_S11_1->setText("%");
		ui->S11_title_label->setText("Reflection:");
		ui->pushButton_unit_S11_1->setChecked(true);
		ui->pushButton_unit_power_forward_1->setChecked(true);
		ui->pushButton_unit_power_reflected_1->setChecked(true);
		ui->pushButton_SWP_unit->setText("Unit: linear");
		ui->pushButton_SWP_unit->setChecked(true);
	}

	emit signal_request_PA_power_update(1);		//0 = logarithmic, 1 = linear;
}



/**********************************************************************************************************************************************************************************
 * MODE BUTTONS
 * *******************************************************************************************************************************************************************************/

//Maybe phase out the CW button entirely?
void GUI_1channel_ODC::on_pushButton_CW_enable_1_clicked()
{
	emit signal_setPWM_enable(1, false);
	emit signal_setALL_enable(1, false);
	emit signal_setDLL_enable(1, false);
}

void GUI_1channel_ODC::on_pushButton_PWM_enable_1_clicked()
{
	emit signal_setPWM_enable(1, ui->pushButton_PWM_enable_1->isChecked());
}

void GUI_1channel_ODC::on_pushButton_ALL_enable_1_clicked()
{
	emit signal_setALL_enable(1, ui->pushButton_ALL_enable_1->isChecked());
}

void GUI_1channel_ODC::on_pushButton_DLL_enable_1_clicked()
{
	emit signal_setDLL_enable(1, ui->pushButton_DLL_enable_1->isChecked());
}

/**********************************************************************************************************************************************************************************
 * REMOTE COMMAND MODE
 * *******************************************************************************************************************************************************************************/
/* TRANSPARENT MODE OFF */
//void GUI_1channel_ODC::on_pushButton_remote_command_OFF_1_clicked()
//{
//	if (remote_command_mode != 0)
//	{
//		remote_command_mode = 0;
//		RCM_controls_enable(true);
//		TCP_disconnect_server();
//		UART_disconnected();

//		/* trigger a reset and clean up the startup junk text */
//		if (config->get_reset_on_startup() == true)
//		{
////			safe_reset();
////			reset_restore();
//			ui->settings_plainTextEdit->appendPlainText(">\tSystem has been reset to defaults.");

//			if (config->get_target_RF_enable() == true)
//			{
//				ui->settings_plainTextEdit->appendPlainText(">\tCaution: RF is auto-enabled");
//			}
//		}
//		else
//		{
//			ui->settings_plainTextEdit->appendPlainText(">\tWarning: Reset is disabled. Settings remain unchanged.\n\tGUI does not accurately reflect the state of the SG Board!");
//		}
//	}
//}

///* ENABLE UART TRANSPARENT MODE */
//void GUI_1channel_ODC::on_pushButton_remote_command_USB_1_clicked()
//{
//	if (remote_command_mode != 1)
//	{
//		/* Stop automated command spam in RCM */
//		readings_timer->stop();

//		if(RCM_port.open(QIODevice::ReadWrite) && SG_port.isOpen())
//		{
//			remote_command_mode = 1;
//			RCM_controls_enable(false);
//			TCP_disconnect_server();
////			connect(readings_timer, SIGNAL(timeout()), this, SLOT(get_PowerInOut()));
//			connect(&RCM_port, SIGNAL(readyRead()), this, SLOT(UART_RCM_Pass()));
//			connect(&SG_port, SIGNAL(readyRead()), this, SLOT(UART_RCM_Return()));
//			ui->settings_plainTextEdit->appendPlainText(QString(">\tUSB connection opened. Device is listening to %1.").arg(RCM_port.portName()));
//		}
//		else
//		{
//			ui->settings_plainTextEdit->appendPlainText(QString(">\tError - unable to open the USB connection:\n\t%1").arg(RCM_port.errorString()));
//			ui->pushButton_remote_command_OFF_1->click(); //indicate that TM did not start.
//		}
//	}
//}

// Live Remote Command.
void GUI_1channel_ODC::on_pushButton_remote_command_OFF_1_clicked()
{
	if (RF_system_constructor->RCM_USB_port->rcm_port->isOpen())
	{
		RF_system_constructor->RCM_USB_port->Disable();
//		RCM_controls_enable(true);

		if (config->get_target_RF_enable() == true)
		{
			ui->settings_plainTextEdit->appendPlainText(">\tCaution: RF is auto-enabled");
			show_notification("Caution: RF is auto-enabled");
		}

		if (config->get_reset_on_startup() == false)
		{
			ui->settings_plainTextEdit->appendPlainText(">\tCaution: Auto-Resetting is disabled. Settings remain unchanged.");
			show_notification("Caution: Auto-Resetting is disabled.");
		}
		else
		{
			emit signal_execute_reset_SGx(1);
		}
	}

	configure_auto_error_clearing();
}

void GUI_1channel_ODC::on_pushButton_remote_command_USB_1_clicked()
{
	if (!RF_system_constructor->RCM_USB_port->rcm_port->isOpen())
	{
		if (!RF_system_constructor->RCM_USB_port->Enable())
		{
			ui->pushButton_remote_command_OFF_1->setChecked(true);
			ui->settings_plainTextEdit->appendPlainText(QString(">\tError - unable to open the USB connection:\n\t%1").arg(RF_system_constructor->RCM_USB_port->rcm_port->errorString()));
		}
		else
		{
//			RCM_controls_enable(false);
			ui->settings_plainTextEdit->appendPlainText(QString(">\tUSB connection opened. Device is listening to %1.").arg(RF_system_constructor->RCM_USB_port->rcm_port->portName()));
		}
	}

	configure_auto_error_clearing();
}

//
// TODO:
// Implement regular RCM mode || Or make live RCM mode fully functional... somehow.
// RF_Channel needs an RCM feature (instead of just a mesage handler)
// Disabling the readings should happen in the RF_Channel, instead of relying on a signal of which you don't know the exact timing.
//

//
// TODO:
// RCM mode is stateless at the start. Needs a handler or something...
//

//void GUI_1channel_ODC::on_pushButton_remote_command_OFF_1_clicked()
//{
//	RF_system_constructor->RCM_USB_port->Disable();

//	disconnect(RF_system_constructor->RCM_USB_port->rcm_port, &QSerialPort::readyRead, RF_system_constructor, &RF_System_1channel_ODC::RCM_USB_message_handler_blind);
//	disconnect(RF_system_constructor, &RF_System_1channel_ODC::signal_RCM_USB_message, RF_System::Channels->at(0), &RF_Channel::RCM_USB_blind_message_handler);
//	disconnect(RF_System::Channels->at(0)->SGX_port->serial, &QSerialPort::readyRead, RF_System::Channels->at(0), &RF_Channel::RCM_USB_blind_response_handler);
//	disconnect(RF_System::Channels->at(0), &RF_Channel::signal_RCM_USB_response, RF_system_constructor, &RF_System_1channel_ODC::RCM_USB_response_handler_blind);

//	emit signal_timerStart(1, RF_system_constructor->config->get_polling_rate_data());
//	emit signal_executeReset(1);
//	RCM_controls_enable(true);
//}

//void GUI_1channel_ODC::on_pushButton_remote_command_USB_1_clicked()
//{
//	RCM_controls_enable(false);
//	emit signal_timerStop(1);

//	QThread::msleep(1000);

//	connect(RF_system_constructor->RCM_USB_port->rcm_port, &QSerialPort::readyRead, RF_system_constructor, &RF_System_1channel_ODC::RCM_USB_message_handler_blind);
//	connect(RF_system_constructor, &RF_System_1channel_ODC::signal_RCM_USB_message, RF_System::Channels->at(0), &RF_Channel::RCM_USB_blind_message_handler);
//	connect(RF_System::Channels->at(0)->SGX_port->serial, &QSerialPort::readyRead, RF_System::Channels->at(0), &RF_Channel::RCM_USB_blind_response_handler);
//	connect(RF_System::Channels->at(0), &RF_Channel::signal_RCM_USB_response, RF_system_constructor, &RF_System_1channel_ODC::RCM_USB_response_handler_blind);

//	if (!RF_system_constructor->RCM_USB_port->rcm_port->isOpen())
//	{
//		if (!RF_system_constructor->RCM_USB_port->Enable())
//		{
//			ui->pushButton_remote_command_OFF_1->setChecked(true);
//		}
//	}
//}

/* ENABLE TCP TRANSPARENT MODE */
void GUI_1channel_ODC::on_pushButton_remote_command_TCP_1_clicked()
{
	/* Cannot have software spamming stuff while in RCM */
//	readings_timer->stop();

	/* Without this check; when TCP button is pressed twice, it complains "the bound adress is already in use"
	 * It then fails to (re)initialize TCP RCM mode and cannot be started anymore afterwards! */
	if(remote_command_mode != 2)
	{
		if(SG_port.isOpen())
		{
			remote_command_mode = 2;
			RCM_controls_enable(false);
			UART_disconnected();

			#warning I think these 'new' declarations could be creating a tiny memory leak. Its virtually inconsequential anyway, but it should be squashed once I figure out how...
			/* Open the TCP server */
			tcpServer = new QTcpServer(this);
			tcpSocket = new QTcpSocket(this);
			if(tcpServer->listen(QHostAddress::Any, 9001))
			{
				/* await a new connection to the IP. */
				connect(tcpServer, SIGNAL(newConnection()), this, SLOT(TCP_newConnection()));
				QList<QHostAddress> ipAddressesList = QNetworkInterface::allAddresses();
				for (int i = 0; i < ipAddressesList.size(); ++i)
				{
					// ToDo: create a network selection prompt. I don't think there's reasonably another way to choose the network.
					/* use the first non-localhost IPv4 address */
					if (ipAddressesList.at(i) != QHostAddress::LocalHost && ipAddressesList.at(i).toIPv4Address())
					{
						ipAddress = ipAddressesList.at(i).toString();
						break;
					}
				}
				if (ipAddress.isEmpty())	// if we did not find one, use IPv4 localhost
				{
					ipAddress = QHostAddress(QHostAddress::LocalHost).toString();
				}
				ui->settings_plainTextEdit->appendPlainText(QString(">\tTCP Server is listening for connections at:\n\tIP:\t\t%1\n\tPort:\t%2").arg(ipAddress).arg(tcpServer->serverPort()));
				qDebug() << "TCP remote command mode ENABLED\nIP:" << ipAddress << "\nport: " << tcpServer->serverPort();
			}
			else
			{
				ui->settings_plainTextEdit->appendPlainText(">\tError - unable to start tcpServer:\n\t" + tcpServer->errorString());
				ui->pushButton_remote_command_OFF_1->click(); //indicate that TM did not start.
				qDebug() << "Error: Unable to start tcpServer: " << tcpServer->errorString();
			}
		}
	}
}

void GUI_1channel_ODC::TCP_newConnection()
{
	tcpSocket = tcpServer->nextPendingConnection();
	if(tcpSocket->state() == QTcpSocket::ConnectedState)
	{
		qDebug() << "New connection established.\n";
		qDebug() << tcpSocket->peerAddress();
		QString peer_address = tcpSocket->peerAddress().toString();
		QStringList peer_address_trim = peer_address.split(":");
		ui->settings_plainTextEdit->appendPlainText(QString(">\tNew connection established with [%1].").arg(peer_address_trim.last()));
	}
	connect(tcpSocket, SIGNAL(disconnected()), this, SLOT(TCP_client_disconnected()));
	connect(tcpSocket, SIGNAL(readyRead()), this, SLOT(TCP_RCM_Pass()));
	connect(&SG_port, SIGNAL(readyRead()), this, SLOT(TCP_RCM_Return()));
}

void GUI_1channel_ODC::TCP_RCM_Pass()
{
	/* close and reopen serial port when sending a message to ensure a $RST didn't break comms */
	SG_port.close();
	SG_port.open(QIODevice::ReadWrite);
	QByteArray input_rx = tcpSocket->readAll();
	SG_port.write(input_rx);
	qDebug() << input_rx.constData();
}

void GUI_1channel_ODC::TCP_RCM_Return()
{
	 QByteArray output_rx = SG_port.readAll();
	 qDebug() << "TCP_RCM_Return -> RX: " << output_rx << "\n";
	 tcpSocket->write(output_rx);
}

/* Disable the TCP server - Shut down TCP RCM mode */
void GUI_1channel_ODC::TCP_disconnect_server()
{
	disconnect(tcpSocket, SIGNAL(disconnected()));
	disconnect(tcpSocket, SIGNAL(readyRead()));
	disconnect(&SG_port, SIGNAL(readyRead()), this, SLOT(TCP_RCM_Return()));

	#warning check if Im not somehow creating a memory leak here... (Does close also delete?)
	if (tcpServer->isListening() || tcpSocket->isOpen()){
		ui->settings_plainTextEdit->appendPlainText(">\tTCP Server has stopped listening for connections.");
		tcpServer->close();
		tcpSocket->close();
		qDebug() << "TCP Remote Command Mode DISABLED";
	}

	SG_port.close();
	SG_port.open(QIODevice::ReadWrite);
	//tcpSocket->deleteLater();

}

void GUI_1channel_ODC::TCP_client_disconnected()
{
	QString peer_address = tcpSocket->peerAddress().toString();
	QStringList peer_address_trim = peer_address.split(":");
	qDebug() << "Client disconnected from the server.";
	if(tcpServer->isListening())
	{
		ui->settings_plainTextEdit->appendPlainText(QString(">\tClient [%1] has disconnected.").arg(peer_address_trim.last()));
		ui->settings_plainTextEdit->appendPlainText(QString(">\tTCP Server is listening for connections at:\n\tIP:\t\t%1\n\tPort:\t%2").arg(ipAddress).arg(tcpServer->serverPort()));
	}
	else
	{
		ui->settings_plainTextEdit->appendPlainText(QString(">\tClient [%1] has been disconnected.").arg(peer_address_trim.last()));
	}
}

/* Close the UART connection - Shut down USB RCM mode */
void GUI_1channel_ODC::UART_disconnected()
{
	disconnect(&RCM_port, SIGNAL(readyRead()), this, SLOT(UART_RCM_Pass()));
	disconnect(&SG_port, SIGNAL(readyRead()), this, SLOT(UART_RCM_Return()));
	if (RCM_port.isOpen())
	{
		RCM_port.close();
		ui->settings_plainTextEdit->appendPlainText(">\tUSB connection closed.");
	}
	/* close and reopen serial port when sending a message to ensure a $RST didn't break comms */
	SG_port.close();
	SG_port.open(QIODevice::ReadWrite);

	QThread::msleep(1000);
	SG_port.clear();

	qDebug() << "USB Remote Command Mode DISABLED";
}

void GUI_1channel_ODC::RCM_controls_enable(bool state)
{
	ui->menu_home_Button->setEnabled(state);
	ui->menu_ALL_Button->setEnabled(state);
	ui->menu_DLL_Button->setEnabled(state);
	ui->menu_help_Button->setEnabled(state);
	ui->menu_settings_Button->setEnabled(state);
	ui->menu_power_Button->setEnabled(state);
	ui->menu_sweep_Button->setEnabled(state);
	ui->menu_PID_Button->setEnabled(state);

//	if(state == false)
//	{
//		ui->menu_sweep_Button->setEnabled(state);
//	}
//	else
//	{
//		ui->menu_sweep_Button->setEnabled(!AIS_enabled);
//	}

//	if(config->get_support_chain_mode() == true)
//	{
//		ui->pushButton_clock_source_standalone_1->setEnabled(state);
//		ui->pushButton_clock_source_master_1->setEnabled(state);
//		ui->pushButton_clock_source_slave_end_1->setEnabled(state);
//		ui->pushButton_clock_source_slave_inline_1->setEnabled(state);
//		ui->label_clock_source->setEnabled(state);
//	}

//	if(config->get_support_PWM_triggering() == true)
//	{
//		ui->pushButton_PWM_triggering_free_running_1->setEnabled(state);
//		ui->pushButton_PWM_triggering_master_1->setEnabled(state);
//		ui->pushButton_PWM_triggering_slave_1->setEnabled(state);
//		ui->PWM_triggering_label->setEnabled(state);
//	}

//	if(config->get_support_AIS_mode() == true)
//	{
//		ui->pushButton_power_control_normal_1->setEnabled(state);
//		ui->pushButton_power_control_analog_1->setEnabled(state);
//		ui->Analog_input_label->setEnabled(state);
//	}

//	if (config->get_support_ext_triggering_mode() == true)
//	{
//		ui->pushButton_external_triggering_OFF_1->setEnabled(state);
//		ui->pushButton_external_triggering_ON_1->setEnabled(state);
//		ui->External_triggering_label->setEnabled(state);
//	}
}

//
//TODO:
//Make it use an argument.
//
void GUI_1channel_ODC::handler_RCM_mode()
{
	if (RCM_supported == true)
	{
		switch(config->get_remote_command_mode())
		{
			case 0:
				ui->pushButton_remote_command_OFF_1->click();
				break;
			case 1:
				ui->menu_settings_Button->click();
				ui->pushButton_remote_command_USB_1->click();
				break;
			case 2:
				ui->menu_settings_Button->click();
				ui->pushButton_remote_command_TCP_1->click();
				break;
			default:
				ui->pushButton_remote_command_OFF_1->click();
				break;
		}
	}
}

/**********************************************************************************************************************************************************************************
 * CHAIN MODE
 * *******************************************************************************************************************************************************************************/
void GUI_1channel_ODC::on_pushButton_clock_source_standalone_1_clicked()
{
	emit signal_setClock_source(1,0);
}

void GUI_1channel_ODC::on_pushButton_clock_source_master_1_clicked()
{
	emit signal_setClock_source(1,1);
}

void GUI_1channel_ODC::on_pushButton_clock_source_slave_end_1_clicked()		//doesn't really get used, should probably be removed altogether.
{
	emit signal_setClock_source(1,2);
}

void GUI_1channel_ODC::on_pushButton_clock_source_slave_inline_1_clicked()
{
	emit signal_setClock_source(1,3);
}

/**********************************************************************************************************************************************************************************
 * Power Control Modes
 * *******************************************************************************************************************************************************************************/
/* Set Regular Power Control Mode (autogain Enabled) */
void GUI_1channel_ODC::on_pushButton_power_control_normal_1_clicked()
{
	emit signal_set_power_control_mode(1, POWER_CONTROL_NORMAL);
}

/* Set AIS mode enable; VGA only; effectively a voltage controlled $GCS */
void GUI_1channel_ODC::on_pushButton_power_control_analog_1_clicked()
{
	emit signal_set_power_control_mode(1, POWER_CONTROL_ANALOG_INPUT);
//	emit signal_setPID_enable(1,false);
}

void GUI_1channel_ODC::on_pushButton_power_control_feedforward_1_clicked()
{
	emit signal_set_power_control_mode(1, POWER_CONTROL_FEED_FWD);
//	emit signal_setPID_enable(1,false);
}

void GUI_1channel_ODC::on_pushButton_power_control_PID_1_clicked()
{
	emit signal_set_power_control_mode(1, POWER_CONTROL_PID);
}


/**********************************************************************************************************************************************************************************
 * External Triggering Mode
 * *******************************************************************************************************************************************************************************/
/* External triggering mode disabled; Enable PWM controls and set duty cycle to 100% */
void GUI_1channel_ODC::on_pushButton_external_triggering_OFF_1_clicked()
{
	emit signal_setExternal_triggering_enable(1, false);
}

/* External triggering mode enabled; Disable PWM controls and set duty cycle to 0% so that we can do external triggering */
void GUI_1channel_ODC::on_pushButton_external_triggering_ON_1_clicked()
{
	emit signal_setExternal_triggering_enable(1, true);
}

/**********************************************************************************************************************************************************************************
 * S11 SWEEP
 * *******************************************************************************************************************************************************************************/
void GUI_1channel_ODC::on_pushButton_SWP_execute_1_clicked()
{
	SWP_enable_GUI(false);
	ui->stackedWidget_3->setCurrentIndex(ui->stackedWidget_3->indexOf(ui->Sweep_graph_page));

	ui->SWP_plot->clearGraphs();
	ui->SWP_plot->replot();

	if (config->get_ODC_workaround_enabled())
	{
		on_pushButton_power_control_feedforward_1_clicked();
	}

	emit signal_execute_SWP(1);
}

void GUI_1channel_ODC::on_pushButton_SWP_execute_2_clicked()
{
	SWP_enable_GUI(false);
	ui->SWP_plot->clearGraphs();
	ui->SWP_plot->replot();

	if (config->get_ODC_workaround_enabled())
	{
		on_pushButton_power_control_feedforward_1_clicked();
	}

	emit signal_execute_SWP(1);
}

void GUI_1channel_ODC::on_pushButton_SWP_unit_clicked()
{
	set_power_reading_units_watt(ui->pushButton_SWP_unit->isChecked());
	SWP_draw_plot(RF_System::Channels->at(0)->SWP_Raw_Data());
}

void GUI_1channel_ODC::on_pushButton_SWP_back_clicked()
{
	ui->pushButton_SWP_execute_1->setChecked(false);
	ui->stackedWidget->setCurrentIndex(ui->stackedWidget->indexOf(ui->sweep_controls_page));
	ui->stackedWidget_2->setCurrentIndex(ui->stackedWidget_2->indexOf(ui->page_logo));
	ui->stackedWidget_3->setCurrentIndex(ui->stackedWidget_3->indexOf(ui->Main_page));
}

void GUI_1channel_ODC::SWP_enable_GUI(bool enable)
{
	enable == true ? ui->pushButton_SWP_execute_1->setText("Sweep") : ui->pushButton_SWP_execute_1->setText("Wait...");
	enable == true ? ui->pushButton_SWP_execute_2->setText("Sweep") : ui->pushButton_SWP_execute_2->setText("Please\nWait...");
	ui->pushButton_SWP_execute_1->setChecked(!enable);
	ui->pushButton_SWP_execute_2->setChecked(!enable);
	ui->pushButton_SWP_back->setEnabled(enable);
	ui->pushButton_SWP_unit->setEnabled(enable);

	for (int i = 0; i < menu_buttonGroup.buttons().count(); i++)
	{
		menu_buttonGroup.buttons().at(i)->setEnabled(enable);
	}
}

/**********************************************************************************************************************************************************************************
 * Overview Page buttons
 * *******************************************************************************************************************************************************************************/
void GUI_1channel_ODC::on_pushButton_go_control_panel_1_clicked()
{
	ui->stackedWidget_5->setCurrentIndex(ui->stackedWidget_5->indexOf(ui->page_control_panel));
	ui->menu_home_Button->click();		//Always start from the Home menu.
}

/**********************************************************************************************************************************************************************************
 * POWER MENU
 * *******************************************************************************************************************************************************************************/
void GUI_1channel_ODC::on_menu_power_Button_clicked()
{
	emit signal_RF_enable(1,false);			//disable RF power because user loses sight of the active data.
	ui->stackedWidget_3->setCurrentIndex(ui->stackedWidget_3->indexOf(ui->power_menu_page));
	last_page_index = ui->stackedWidget_3->currentIndex();
}

void GUI_1channel_ODC::on_pushButton_shutdown_controller_1_clicked()
{
	power_menu_action = 1;
	ui->power_menu_confirm_label->setText("Are you sure you want to shut down?");
	ui->stackedWidget_3->setCurrentIndex(ui->stackedWidget_3->indexOf(ui->power_menu_confirm_page));
}

void GUI_1channel_ODC::on_pushButton_reboot_controller_1_clicked()
{
	power_menu_action = 2;
	ui->power_menu_confirm_label->setText("Are you sure you want to reboot?");
	ui->stackedWidget_3->setCurrentIndex(ui->stackedWidget_3->indexOf(ui->power_menu_confirm_page));
}

void GUI_1channel_ODC::on_pushButton_exit_program_1_clicked()
{
	power_menu_action = 3;
	ui->power_menu_confirm_label->setText("Are you sure you want to exit the program?");
	ui->stackedWidget_3->setCurrentIndex(ui->stackedWidget_3->indexOf(ui->power_menu_confirm_page));
}

void GUI_1channel_ODC::on_pushButton_go_overview_1_clicked()
{
	power_menu_action = 4;
	ui->power_menu_confirm_label->setText("Return to the overview screen?");
	ui->stackedWidget_3->setCurrentIndex(ui->stackedWidget_3->indexOf(ui->power_menu_confirm_page));
}

void GUI_1channel_ODC::on_pushButton_reset_SGx_1_clicked()
{
	power_menu_action = 5;
	ui->power_menu_confirm_label->setText("Are you sure you want to reset the SGx board?");
	ui->stackedWidget_3->setCurrentIndex(ui->stackedWidget_3->indexOf(ui->power_menu_confirm_page));
}

void GUI_1channel_ODC::on_pushButton_reset_protection_1_clicked()
{
	power_menu_action = 6;
	ui->power_menu_confirm_label->setText("Are you sure you want to reset the Protection board?");
	ui->stackedWidget_3->setCurrentIndex(ui->stackedWidget_3->indexOf(ui->power_menu_confirm_page));
}

void GUI_1channel_ODC::on_pushButton_restart_program_1_clicked()
{
	power_menu_action = 7;
	ui->power_menu_confirm_label->setText("Are you sure you want to restart the program?");
	ui->stackedWidget_3->setCurrentIndex(ui->stackedWidget_3->indexOf(ui->power_menu_confirm_page));
}

void GUI_1channel_ODC::on_pushButton_clear_errors_1_clicked()
{
	emit signal_execute_error_clear(1);
}

//
// TODO: I wish I could just combine this with the destructor and get rid of the extra code...
// Maybe I could trigger a delayed shutdown and then activate the destructor.
//
void GUI_1channel_ODC::on_Exit_Yes_button_clicked()
{
	if (remote_command_mode > 0){
		ui->pushButton_remote_command_OFF_1->click();	//enable control again.
	}
	QProcess process;

	while(RF_enabled != false)
	{
		serial_setRF_enable(SG_port,channel_select,false);
	}
	switch(power_menu_action)
	{
		case 1:
			#if defined(Q_OS_LINUX)
			process.startDetached("sudo poweroff -f");
			this->close();
			#elif defined (Q_OS_WINDOWS)
			qDebug() << "Feature not supported on windows.";
			#endif
			break;
		case 2:
			#if defined(Q_OS_LINUX)
			process.startDetached("sudo reboot -f");
			this->close();
			#elif defined (Q_OS_WINDOWS)
			qDebug() << "Feature not supported on windows.";
			#endif
			break;
		case 3:
			this->close();
			break;
		case 4:
			ui->stackedWidget_5->setCurrentIndex(ui->stackedWidget_5->indexOf(ui->page_overview));
			break;
		case 5:
			emit signal_execute_reset_SGx(1);
			ui->menu_home_Button->click();	//Always start from the Home menu.
			break;
		case 6:
			emit signal_execute_reset_protection(1);
			ui->stackedWidget_3->setCurrentIndex(last_page_index);
			break;
		case 7:
			handler_execute_restart_program();
		default:
			break;
	}
}

void GUI_1channel_ODC::on_Exit_No_button_clicked()
{
	ui->stackedWidget_3->setCurrentIndex(last_page_index);
}

void GUI_1channel_ODC::handler_execute_restart_program()
{
	QProcess process;
	process.startDetached(QCoreApplication::applicationDirPath() + "/" + QCoreApplication::applicationName());
	this->close();
}



/**********************************************************************************************************************************************************************************
 * Channel reading handlers
 * *******************************************************************************************************************************************************************************/
//Miscellaneous
void GUI_1channel_ODC::handler_channel_working(int channel_number, bool enabled)
{
	this->setEnabled(enabled);
}

void GUI_1channel_ODC::handler_channel_init_failed(int intrasys_num)
{
	qDebug() << "GUI: Subsystem[" << intrasys_num << "] Initialization failed.";
	if (RF_system_constructor->config->get_init_failed_exit() == true)
	{
		qDebug() << "Exiting GUI.";
		this->close();
	}
}

void GUI_1channel_ODC::handler_interlock_enable_get(bool enable)	//3V3 OK signal on the raspberry Pi
{
	ui->label_interlock_1->setEnabled(enable);
	ui->label_interlock_2->setEnabled(enable);
	if (enable)
	{
		ui->label_interlock_2->setText("✔");
	}
	else
	{
		ui->label_interlock_2->setText("✖");
	}
}

void GUI_1channel_ODC::handler_firmware_version_get(int intrasys_num, int channel_num, QString version)
{
	if (version != "")
	{
		firmware_version_check(version);
		ui->firmware_version_label_2->setText(version);
	}
}

void GUI_1channel_ODC::handler_error_get(int intrasys_num, quint64 error, QStringList error_messages)
{
	if (error > 0)
	{
		//display in error label.
		ui->label_error_2->setText(QString("0x%1").arg(error,0,16));

		QString message_string = "";
		for (int i = 0; i < error_messages.count(); i++)
		{
			if (error_messages.at(i) != "")
			{
				message_string += error_messages.at(i);
			}
			if (i != error_messages.count() - 1)
			{
				message_string += "\n";
			}
		}

		show_error(message_string);

		//overview page
		ui->label_status_4->setText(message_string);
	}
	else
	{
		ui->label_error_2->setText(error_messages.at(0));
		close_error();

		//overview page
		ui->label_status_4->setText(error_messages.at(0));
	}

	//overview page
	ui->label_status_2->setText(QString("0x%1").arg(error,0,16));

	//datalogging
	datalogger->setData(9, QString::number(error,16));
}

void GUI_1channel_ODC::handler_reset_detected(int intrasys_num, int channel_number)
{
	show_notification("A reset has occured");
	ui->settings_plainTextEdit->appendPlainText(">\tSystem has been reset to defaults.");
}

void GUI_1channel_ODC::handler_temperature_get(int intrasys_num, double temperature)
{
	ui->label_temperature_2->setText(QString::number(temperature,'f',1));

	//overview page
	ui->label_temperature_5->setText(QString::number(temperature,'f',1));

	//datalogging
	datalogger->setData(8, QString::number(temperature));
}

#ifdef I2C_TEMP_SENSOR
void GUI_1channel_ODC::handler_sensor_temperature_get(double val)
{
	ui->label_temperature_2->setText(QString::number(val,'f',1));
	//overview page
	ui->label_temperature_5->setText(QString::number(val,'f',1));
	datalogger->setData(8, QString::number(val));
}
#endif

void GUI_1channel_ODC::handler_power_control_mode_get(int intrasys_num, int mode)
{
	//
	//TODO:
	//It's not very nice that the controls are being set over and over and over again...
	//
	switch(mode)
	{
		case POWER_CONTROL_NORMAL:
			show_autogain_controls(intrasys_num, true);
			show_GCS_controls(intrasys_num, false);
			show_MCS_controls(intrasys_num, false);
			ui->pushButton_power_control_normal_1->setChecked(true);
			break;
		case POWER_CONTROL_ANALOG_INPUT:
			show_autogain_controls(intrasys_num, false);
			show_GCS_controls(intrasys_num, true);
			show_MCS_controls(intrasys_num, true);
			ui->pushButton_power_control_analog_1->setChecked(true);
			break;
		case POWER_CONTROL_FEED_FWD:
			show_autogain_controls(intrasys_num, true);
			show_GCS_controls(intrasys_num, true);
			show_MCS_controls(intrasys_num, true);
			ui->pushButton_power_control_feedforward_1->setChecked(true);
			break;
		case POWER_CONTROL_AMPLIFIER:
			//
			//TODO:
			//Amplifier Mode
			//
			break;
		case POWER_CONTROL_PID:
			show_autogain_controls(intrasys_num, false);
			show_GCS_controls(intrasys_num, false);
			show_MCS_controls(intrasys_num, false);
			ui->pushButton_power_control_PID_1->setChecked(true);
			break;
		case POWER_CONTROL_UNKNOWN:
			ui->pushButton_power_control_normal_1->setChecked(false);
			ui->pushButton_power_control_analog_1->setChecked(false);
			ui->pushButton_power_control_feedforward_1->setChecked(false);
			ui->pushButton_power_control_PID_1->setChecked(false);
			show_autogain_controls(intrasys_num, false);
			show_GCS_controls(intrasys_num, false);
			show_MCS_controls(intrasys_num, false);
//			show_feedforward_controls(intrasys_num, false);
			break;
		default:
			break;
	}
}

void GUI_1channel_ODC::handler_RF_enable_get(int intrasys_num, bool RF_enabled)
{
	if (RF_enabled == true)
	{
		ui->pushButton_RF_enable_1->setText("RF is ON");
		datalogger->setData(0, "true");

		//overview page
		ui->label_RF_enable_1->setText("RF is ON");

	}
	else
	{
		ui->pushButton_RF_enable_1->setText("RF is OFF");
		datalogger->setData(0, "false");

		//overview page
		ui->label_RF_enable_1->setText("RF is OFF");
	}

	ui->pushButton_RF_enable_1->setChecked(RF_enabled);
}

void GUI_1channel_ODC::handler_PA_power_readings(int intrasys_num, double PA_power_fwd_dbm, double PA_power_rfl_dbm, double PA_s11, double PA_power_fwd_watt, double PA_power_rfl_watt, double PA_s11_reflection)
{
	double PA_power_minimum_dbm = 30;

	if (ui->pushButton_unit_power_forward_1->isChecked())
	{
		ui->label_power_forward->setText(QString::number(PA_power_fwd_watt,'f',1));
	}
	else
	{
		ui->label_power_forward->setText(QString::number(PA_power_fwd_dbm,'f',1));
	}

	if (ui->pushButton_unit_power_reflected_1->isChecked())
	{
		ui->label_power_reflected->setText(QString::number(PA_power_rfl_watt,'f',1));
	}
	else
	{
		ui->label_power_reflected->setText(QString::number(PA_power_rfl_dbm,'f',1));
	}

	if (PA_power_fwd_dbm >= PA_power_minimum_dbm)
	{
		if (ui->pushButton_unit_S11_1->isChecked())
		{
			ui->label_s11_value->setText(QString::number(PA_s11_reflection,'f',1));
		}
		else
		{
			ui->label_s11_value->setText(QString::number(PA_s11,'f',1));
		}

		//overview page
		ui->label_S11_2->setText(QString::number(PA_s11_reflection,'f',1));
	}
	else
	{
		ui->label_s11_value->setText("-");

		//overview page
		ui->label_S11_2->setText("-");
	}

	//overview page
	ui->label_PA_power_forward_2->setText(QString::number(PA_power_fwd_watt,'f',1));
	ui->label_PA_power_reflected_2->setText(QString::number(PA_power_rfl_watt,'f',1));

	//datalogging
	datalogger->setData(3, QString::number(PA_power_fwd_watt,'f',1));
	datalogger->setData(4, QString::number(PA_power_rfl_watt,'f',1));
	datalogger->setData(5, QString::number(PA_s11_reflection,'f',1));
}

void GUI_1channel_ODC::handler_power_get(int intrasys_num, double power_dbm, double power_watt)
{
	ui->pushButton_power_dbm_1->setText(zeroChopper(QString::number(power_dbm, 'f', 2)));
	ui->pushButton_power_watt_1->setText(zeroChopper(QString::number(power_watt, 'f', 2)));
	ui->pushButton_power_dbm_2->setText(zeroChopper(QString::number(power_dbm, 'f', 2)));
	ui->pushButton_power_watt_2->setText(zeroChopper(QString::number(power_watt, 'f', 2)));

	//overview page
	ui->label_PA_power_setpoint_2->setText(zeroChopper(QString::number(power_watt, 'f', 2)));

	//datalogging
	datalogger->setData(2, QString::number(power_watt));
}

void GUI_1channel_ODC::handler_VGA_attenuation_get(int intrasys_num, double attenuation)
{
	ui->pushButton_VGA_attenuation_1->setText(zeroChopper(QString::number(attenuation, 'f', 1)));
}

void GUI_1channel_ODC::handler_IQMod_magnitude_get(int intrasys_num, double magnitude)
{
	ui->pushButton_IQMod_magnitude_1->setText(zeroChopper(QString::number(magnitude, 'f', 1)));
}

void GUI_1channel_ODC::handler_clock_source_get(int intrasys_num, int mode)
{
	(void) intrasys_num;

	/* protection against out of index crash if a mode is configured that does not have a corresponding button. */
	bool valid_clock_source_mode = (mode < clock_source_buttonGroup.buttons().count());

	if (valid_clock_source_mode == true)
	{
		if (clock_source_buttonGroup.buttons().at(mode)->isChecked() == false)
		{
			clock_source_buttonGroup.buttons().at(mode)->setChecked(true);
		}
	}
	else
	{
		qDebug() << "Unknown clock source mode: " << mode;
		return;
	}
}

//void GUI_1channel_ODC::handler_external_triggering_enable_get(int intrasys_num, bool enable)
//{
//	enable == true ? ui->pushButton_external_triggering_ON_1->setChecked(true) : ui->pushButton_external_triggering_OFF_1->setChecked(true);
//}


//CW
void GUI_1channel_ODC::handler_frequency_get(int intrasys_num, double frequency_mhz)
{
	ui->pushButton_frequency_1->setText(zeroChopper(QString::number(frequency_mhz, 'f', 3)));

	//DLL lock frequency indication with flexible precision, to avoid annoying jumpiness in the UI
	QString string1 = zeroChopper(QString::number(frequency_mhz,'f',3));
	QString string2 = ui->pushButton_DLL_step_frequency_2->text();
	int lock_precision = 0;

	if (string1.contains("."))
	{
		lock_precision = string1.count() - string1.indexOf(".") - 1;
	}
	if (string2.contains("."))
	{
		int temp_precision = string2.count() - string2.indexOf(".") - 1;
		if (temp_precision > lock_precision)
		{
			lock_precision = temp_precision;
		}
	}

	ui->label_DLL_frequency_lock_2->setText(QString::number(frequency_mhz, 'f', lock_precision));

	//overview page
	ui->label_frequency_2->setText(zeroChopper(QString::number(frequency_mhz, 'f', 3)));

	//datalogging
	datalogger->setData(1, QString::number(frequency_mhz));
}

void GUI_1channel_ODC::handler_phase_get(int intrasys_num, double phase_degrees)
{
	ui->pushButton_phase_1->setText(zeroChopper(QString::number(phase_degrees, 'f', 1 )));
}

void GUI_1channel_ODC::handler_CW_enable_get(int intrasys_num, bool enable)
{
	ui->pushButton_CW_enable_1->setChecked(enable);
}

//PWM
void GUI_1channel_ODC::handler_PWM_settings_get(int intrasys_num, double frequency_hz, int duty_cycle)
{
	if (duty_cycle <= 0)							//PWM disabled <> External Triggering Mode enabled.
	{
		ui->pushButton_external_triggering_ON_1->setChecked(true);
		ui->pushButton_PWM_enable_1->setChecked(false);
		show_PWM_settings(false);
		ui->pushButton_SWP_execute_1->setEnabled(false);
		ui->pushButton_SWP_execute_2->setEnabled(false);
		ui->pushButton_PWM_enable_1->setEnabled(false);

		//overview page
		ui->label_PWM_enable_1->setText("PWM is OFF");
	}
	else if (duty_cycle >= 1 && duty_cycle <= 99)	//PWM enabled <> External Triggering Mode disabled.
	{
		ui->pushButton_external_triggering_OFF_1->setChecked(true);
		ui->pushButton_PWM_enable_1->setChecked(true);
		show_PWM_settings(true);
		ui->pushButton_PWM_enable_1->setEnabled(true);
		ui->pushButton_SWP_execute_1->setEnabled(true);
		ui->pushButton_SWP_execute_2->setEnabled(true);

		//overview page
		ui->label_PWM_enable_1->setText("PWM is ON");
	}
	else											//PWM disabled <> External Triggering Mode disabled.
	{
		ui->pushButton_external_triggering_OFF_1->setChecked(true);
		ui->pushButton_PWM_enable_1->setChecked(false);
		show_PWM_settings(false);
		ui->pushButton_PWM_enable_1->setEnabled(true);
		ui->pushButton_SWP_execute_1->setEnabled(true);
		ui->pushButton_SWP_execute_2->setEnabled(true);

		//overview page
		ui->label_PWM_enable_1->setText("PWM is OFF");
	}

	ui->pushButton_PWM_duty_cycle_1->setText(zeroChopper(QString::number(duty_cycle,'f',1)));
	ui->pushButton_PWM_frequency_1->setText(zeroChopper(QString::number(frequency_hz, 'f', 2)));

	//overview page
	ui->label_PWM_duty_cycle_2->setText(zeroChopper(QString::number(duty_cycle, 'f', 1)));
	ui->label_PWM_frequency_2->setText(zeroChopper(QString::number(frequency_hz, 'f', 2)));

	//datalogging
	datalogger->setData(6, QString::number(frequency_hz));
	datalogger->setData(7, QString::number(duty_cycle));
}

//DLL
void GUI_1channel_ODC::handler_DLL_enable_get(int intrasys_num, bool enable)
{
	ui->pushButton_DLL_enable_1->setChecked(enable);
	show_DLL_settings(enable);

	//overview page
	enable == true ? ui->label_DLL_enable_1->setText("DLL is ON") : ui->label_DLL_enable_1->setText("DLL is OFF");
}

void GUI_1channel_ODC::handler_DLL_settings_get(int intrasys_num, double frequency_lower, double frequency_upper, double frequency_start, double frequency_step, double threshold, double main_delay)
{
	ui->pushButton_DLL_frequency_limit_lower_1->setText(zeroChopper(QString::number(frequency_lower, 'f', 3)));
	ui->pushButton_DLL_frequency_limit_upper_1->setText(zeroChopper(QString::number(frequency_upper, 'f', 3)));
	ui->pushButton_DLL_start_frequency_1->setText(zeroChopper(QString::number(frequency_start, 'f', 3)));
	ui->pushButton_DLL_step_frequency_1->setText(zeroChopper(QString::number(frequency_step, 'f', 3)));
	ui->pushButton_DLL_step_frequency_2->setText(zeroChopper(QString::number(frequency_step, 'f', 3)));
	ui->pushButton_DLL_threshold_1->setText(zeroChopper(QString::number(threshold, 'f', 2)));
	ui->pushButton_DLL_threshold_2->setText(zeroChopper(QString::number(threshold, 'f', 2)));
	ui->pushButton_DLL_main_delay_1->setText(zeroChopper(QString::number(main_delay, 'f', 1)));

	//overview page
	ui->label_DLL_frequency_lower_2->setText(zeroChopper(QString::number(frequency_lower, 'f', 3)));
	ui->label_DLL_frequency_upper_2->setText(zeroChopper(QString::number(frequency_upper, 'f', 3)));
	ui->label_DLL_frequency_step_2->setText(zeroChopper(QString::number(frequency_step, 'f', 3)));
	ui->label_DLL_threshold_2->setText(zeroChopper(QString::number(threshold, 'f', 2)));
}

//ALL
void GUI_1channel_ODC::handler_ALL_compatible_get(int intrasys_num, bool compatibility)
{
	if (compatibility == true)
	{
		show_ALL_settings(config->get_menu_ALL_enabled());
		ui->menu_ALL_Button->setVisible(config->get_menu_ALL_enabled());					//enable or disable the ALL menu button
		ui->menu_ALL_Button->setEnabled(config->get_menu_ALL_enabled());
		ui->pushButton_ALL_enable_1->setVisible(config->get_controls_ALL_enabled());		//enable or disable the ALL enable button
		ui->pushButton_ALL_enable_1->setEnabled(config->get_controls_ALL_enabled());
	}
	else
	{
		//
		//TODO:
		//visibility disabled settings can still be bypassed by start-up state / RCM on SG4A's
		//
		show_ALL_settings(false);
		ui->menu_ALL_Button->setVisible(false);				//enable or disable the ALL menu button
		ui->menu_ALL_Button->setEnabled(false);
		ui->pushButton_ALL_enable_1->setVisible(false);		//enable or disable the ALL enable button
		ui->pushButton_ALL_enable_1->setEnabled(false);
	}
}

void GUI_1channel_ODC::handler_ALL_enable_get(int intrasys_num, bool enable)
{
	ui->pushButton_ALL_enable_1->setChecked(enable);
	show_ALL_settings(enable);

	//
	//TODO:
	//ALL Compatibility missing...
	//

	ui->pushButton_DVGA_attenuation_forward_1->setEnabled(enable);
	ui->pushButton_DVGA_attenuation_reflected_1->setEnabled(enable);
	ui->label_DVGA_attenuation_forward_1->setEnabled(enable);
	ui->label_DVGA_amplifier_forward_1->setEnabled(enable);
	ui->pushButton_DVGA_amplifier_forward_1->setEnabled(enable);
	ui->pushButton_DVGA_amplifier_reflected_1->setEnabled(enable);
	ui->label_DVGA_attenuation_reflected_1->setEnabled(enable);
	ui->label_DVGA_amplifier_reflected_1->setEnabled(enable);
}

void GUI_1channel_ODC::handler_ALL_settings_get(int intrasys_num, double frequency_lower_voltage, double frequency_upper_voltage, double threshold_voltage)
{
	ui->pushButton_ALL_frequency_limit_lower_1->setText(zeroChopper(QString::number(frequency_lower_voltage, 'f', 2)));
	ui->pushButton_ALL_frequency_limit_upper_1->setText(zeroChopper(QString::number(frequency_upper_voltage, 'f', 2)));
	ui->pushButton_ALL_threshold_1->setText(zeroChopper(QString::number(threshold_voltage, 'f', 2)));
	ui->pushButton_ALL_threshold_2->setText(zeroChopper(QString::number(threshold_voltage, 'f', 2)));
}

//DVGA's
void GUI_1channel_ODC::handler_DVGA_compatible_get(int intrasys_num, bool compatibility)
{
	ui->label_DVGA_attenuation_forward_1->setVisible(compatibility);
	ui->label_DVGA_attenuation_reflected_1->setVisible(compatibility);
	ui->label_DVGA_amplifier_forward_1->setVisible(compatibility);
	ui->label_DVGA_amplifier_reflected_1->setVisible(compatibility);
	ui->pushButton_DVGA_attenuation_forward_1->setVisible(compatibility);
	ui->pushButton_DVGA_attenuation_reflected_1->setVisible(compatibility);
	ui->pushButton_DVGA_amplifier_forward_1->setVisible(compatibility);
	ui->pushButton_DVGA_amplifier_reflected_1->setVisible(compatibility);
}

void GUI_1channel_ODC::handler_DVGA_forward_settings_get(int intrasys_num, bool amplifier_enable, double attenuation)
{
	ui->pushButton_DVGA_attenuation_forward_1->setText(zeroChopper(QString::number(attenuation, 'f', 2)));
	if(amplifier_enable == true)
	{
		ui->pushButton_DVGA_amplifier_forward_1->setText("ON");
	}
	else
	{
		ui->pushButton_DVGA_amplifier_forward_1->setText("OFF");
	}

	ui->pushButton_DVGA_amplifier_forward_1->setChecked(amplifier_enable);
}

void GUI_1channel_ODC::handler_DVGA_reflected_settings_get(int intrasys_num, bool amplifier_enable, double attenuation)
{
	ui->pushButton_DVGA_attenuation_reflected_1->setText(zeroChopper(QString::number(attenuation, 'f', 2)));
	if(amplifier_enable == true)
	{
		ui->pushButton_DVGA_amplifier_reflected_1->setText("ON");
	}
	else
	{
		ui->pushButton_DVGA_amplifier_reflected_1->setText("OFF");
	}

	ui->pushButton_DVGA_amplifier_reflected_1->setChecked(amplifier_enable);
}

//PID
void GUI_1channel_ODC::handler_PID_settings_get(int intrasys_num, double Kp, double Ki, double Kd, double setpoint, double scaling, double offset)
{
	ui->pushButton_PID_kp_1->setText(zeroChopper(QString::number(Kp, 'f', 2)));
	ui->pushButton_PID_ki_1->setText(zeroChopper(QString::number(Ki, 'f', 2)));
	ui->pushButton_PID_kd_1->setText(zeroChopper(QString::number(Kd, 'f', 2)));
	ui->pushButton_PID_setpoint_1->setText(zeroChopper(QString::number(setpoint, 'f', 2)));
	ui->pushButton_PID_scaling_1->setText(zeroChopper(QString::number(scaling, 'f', 2)));
	ui->pushButton_PID_offset_1->setText(zeroChopper(QString::number(offset, 'f', 2)));
}

//SWP
void GUI_1channel_ODC::handler_SWP_settings_get(int instrasys_num, double frequency_start, double frequency_stop, double frequency_step, double power_dbm, double power_watt)
{
	ui->pushButton_SWP_start_frequency_1->setText(zeroChopper(QString::number(frequency_start, 'f', 3)));
	ui->pushButton_SWP_stop_frequency_1->setText(zeroChopper(QString::number(frequency_stop, 'f', 3)));
	ui->pushButton_SWP_step_frequency_1->setText(zeroChopper(QString::number(frequency_step, 'f', 3)));
	ui->pushButton_SWP_power_dbm_1->setText(zeroChopper(QString::number(power_dbm, 'f', 1)));
	ui->pushButton_SWP_power_watt_1->setText(zeroChopper(QString::number(power_watt, 'f', 1)));
}

void GUI_1channel_ODC::handler_SWP_measurement_get(int intrasys_num, QString SWP_raw_data)
{
	if(SWP_raw_data.contains("$SWPD,") && SWP_raw_data.contains("OK\r\n"))
	{
		SWP_draw_plot(SWP_raw_data);
	}
	else
	{
		ui->stackedWidget_3->setCurrentIndex(ui->stackedWidget_3->indexOf(ui->Main_page));
		if (SWP_raw_data != "")
		{
			QStringList response = SWP_raw_data.split(",");
			if (response.count() >= 3)
			{
				QString error = response.at(2);
				error.chop(2);
				show_notification("Sweep returned error: " + error);
			}
		}
		else
		{
			show_notification("Sweep timed out.");
		}
	}

	SWP_enable_GUI(true);
}

void GUI_1channel_ODC::SWP_draw_plot(QString SWP_raw_data)
{
	QStringList SWP_data = SWP_raw_data.split("\r\n");

	//Remove the OK entry the empty one that comes after.
	QVector<double>	SWP_freq_data;
	QVector<double>	SWP_frw_data;
	QVector<double>	SWP_rfl_data;
	QVector<double>	SWP_s11_dbm_data;
	QVector<double>	SWP_s11_watt_data;

	SWP_freq_data.resize(SWP_data.count()-2);
	SWP_frw_data.resize(SWP_data.count()-2);
	SWP_rfl_data.resize(SWP_data.count()-2);
	SWP_s11_dbm_data.resize(SWP_data.count()-2);
	SWP_s11_watt_data.resize(SWP_data.count()-2);

	QRegExp regexp("\\$SWP\\D?,\\d+,(\\d+.?\\d+),(-?\\d+.?\\d+),(-?\\d+.?\\d+)");

	for (int i = 0; i < SWP_data.count() - 2; i++)
	{
		 qDebug() << SWP_data.at(i) << "#" << i;
		 if (regexp.indexIn(SWP_data.at(i))>=0)
		 {
			  QString string1 = regexp.cap(1);
			  QString string2 = regexp.cap(2);
			  QString string3 = regexp.cap(3);

			  SWP_freq_data[i] = string1.toDouble();
			  SWP_frw_data[i] = string2.toDouble();
			  SWP_rfl_data[i] = string3.toDouble();

			  SWP_s11_dbm_data[i] = SWP_rfl_data[i] - SWP_frw_data[i];
			  SWP_s11_watt_data[i] = (convert_dbm_to_watt(SWP_rfl_data[i]) / convert_dbm_to_watt(SWP_frw_data[i])) * 100;
			  /* Perhaps theoretically a division by zero is possible here, but the minimum power value that can be set is 0.1W / 20dBm anyway...
			   * Since we get our power values in dBm and convert to watt, there's no such thing as 0 dbm anyway... therefor in practice divide by zero doesn't occur. */
		 }
	}

	ui->SWP_plot->addGraph();
	ui->SWP_plot->xAxis->setLabel("Frequency (MHz)");
	ui->SWP_plot->xAxis->setRange(SWP_freq_data[0],SWP_freq_data[SWP_freq_data.size()-1]);
	ui->SWP_plot->yAxis->setNumberFormat("f");
	ui->SWP_plot->yAxis->setNumberPrecision(2);
	double min_val, max_val;

	if(!ui->pushButton_SWP_unit->isChecked())	//dbm
	{
		ui->SWP_plot->graph(0)->setData(SWP_freq_data,SWP_s11_dbm_data);

		min_val = *std::min_element(SWP_s11_dbm_data.constBegin(),SWP_s11_dbm_data.constEnd());
		if (min_val > 0){
			 min_val = 0;
		}
		max_val = *std::max_element(SWP_s11_dbm_data.constBegin(),SWP_s11_dbm_data.constEnd());
		if(max_val < 0){
			 max_val = 0;
		}

		ui->SWP_plot->yAxis->setRange(min_val*1.1, max_val*1.1);
		ui->SWP_plot->yAxis->setLabel("S11 (dB)");
		//ui->SWP_plot->yAxis->setRangeReversed(true);
	}
	else
	{
		ui->SWP_plot->graph(0)->setData(SWP_freq_data,SWP_s11_watt_data);

		max_val = *std::max_element(SWP_s11_watt_data.constBegin(),SWP_s11_watt_data.constEnd());

		if(max_val <= 1)
		{
			max_val = 1;
		}
		ui->SWP_plot->yAxis->setRange(0,max_val);
		ui->SWP_plot->yAxis->setLabel("Reflection (%)");
	}

	ui->SWP_plot->setInteractions(QCP::iRangeDrag | QCP::iRangeZoom | QCP::iSelectItems);
	ui->SWP_plot->replot();
}


//PSU
void GUI_1channel_ODC::handler_PSU_IU_get(int intrasys_num, int psu_num, double voltage, double current, double power)
{
	switch (psu_num)
	{
		case 0:
			//overview page
			ui->label_PSU1_voltage_1->setText(QString::number(voltage,'f',1));
			ui->label_PSU1_current_1->setText(QString::number(current,'f',1));
			ui->label_PSU1_power_1->setText(QString::number(power,'f',0));
			//PSU page
			ui->label_PSU1_voltage_3->setText(QString::number(voltage,'f',1));
			ui->label_PSU1_current_3->setText(QString::number(current,'f',1));
			ui->label_PSU1_power_3->setText(QString::number(power,'f',0));
			break;
		case 1:
			//overview page
			ui->label_PSU2_voltage_1->setText(QString::number(voltage,'f',1));
			ui->label_PSU2_current_1->setText(QString::number(current,'f',1));
			ui->label_PSU2_power_1->setText(QString::number(power,'f',0));
			//PSU page
			ui->label_PSU2_voltage_3->setText(QString::number(voltage,'f',1));
			ui->label_PSU2_current_3->setText(QString::number(current,'f',1));
			ui->label_PSU2_power_3->setText(QString::number(power,'f',0));
			break;
		case 2:
			//overview page
			ui->label_PSU3_voltage_1->setText(QString::number(voltage,'f',1));
			ui->label_PSU3_current_1->setText(QString::number(current,'f',1));
			ui->label_PSU3_power_1->setText(QString::number(power,'f',0));
			//PSU page
			ui->label_PSU3_voltage_3->setText(QString::number(voltage,'f',1));
			ui->label_PSU3_current_3->setText(QString::number(current,'f',1));
			ui->label_PSU3_power_3->setText(QString::number(power,'f',0));
			break;
		case 3:
			//overview page
			ui->label_PSU4_voltage_1->setText(QString::number(voltage,'f',1));
			ui->label_PSU4_current_1->setText(QString::number(current,'f',1));
			ui->label_PSU4_power_1->setText(QString::number(power,'f',0));
			//PSU page
			ui->label_PSU4_voltage_3->setText(QString::number(voltage,'f',1));
			ui->label_PSU4_current_3->setText(QString::number(current,'f',1));
			ui->label_PSU4_power_3->setText(QString::number(power,'f',0));
			break;
		default:
			break;
	}

	datalogger->setData("PSU_Voltage_" + QString::number(psu_num + 1), QString::number(voltage,'f',1));
	datalogger->setData("PSU_Current_" + QString::number(psu_num + 1), QString::number(current,'f',1));
	datalogger->setData("PSU_Power_" + QString::number(psu_num + 1), QString::number(power,'f',1));
}

void GUI_1channel_ODC::handler_PSU_power_efficiency_get(int intrasys_num, double val)
{
	ui->label_PSU_efficiency_2->setText(QString::number(val, 'f',1));
	ui->label_PSU_efficiency_5->setText(QString::number(val, 'f',1));
	datalogger->setData("PSU_Power_Efficiency", QString::number(val, 'f',1));
}

void GUI_1channel_ODC::handler_PSU_voltage_setpoint_get(int intrasys_num, double val)
{
	if (val >= 0)
	{
		ui->pushButton_PSU_voltage_1->setText(zeroChopper(QString::number(val,'f',1)));
	}
	else
	{
		ui->pushButton_PSU_voltage_1->setText("-");
	}
}

void GUI_1channel_ODC::handler_PSU_current_limit_get(int intrasys_num, double val)
{
	if (val >= 0)
	{
		ui->pushButton_PSU_current_limit_1->setText(zeroChopper(QString::number(val,'f',1)));
	}
	else
	{
		ui->pushButton_PSU_current_limit_1->setText("-");
	}
}

void GUI_1channel_ODC::handler_PSU_enable_combined_get(int intrasys_num, bool enable)
{
	//
	//TODO:
	//PSU enable state change is not very elegant.
	//All PSU's are shown as enabled; even though there may be less present.
	//Data logging trigger is kind of... indirect?
	//

	QString enable_string = "";
	enable == true ? enable_string = "ON" : enable_string = "OFF";

	//Enable/Disable datalogging when PSU enable state changes;
	if (ui->pushButton_PSU_enable_1->text() != enable_string)
	{
		datalogger->enable_datalogging(enable);
	}

	//overview page
	ui->label_PSU1_enable_1->setText(enable_string);
	ui->label_PSU2_enable_1->setText(enable_string);
	ui->label_PSU3_enable_1->setText(enable_string);
	ui->label_PSU4_enable_1->setText(enable_string);

	//PSU page
	ui->pushButton_PSU_enable_1->setText(enable_string);
	ui->pushButton_PSU_enable_1->setChecked(enable);
	ui->label_PSU1_enable_2->setText(enable_string);
	ui->label_PSU2_enable_2->setText(enable_string);
	ui->label_PSU3_enable_2->setText(enable_string);
	ui->label_PSU4_enable_2->setText(enable_string);
}

void GUI_1channel_ODC::handler_PSU_dissipation_get(int intrasys_num, double val)
{
	ui->label_PSU_dissipation_2->setText(zeroChopper(QString::number(val,'f',1)));
	ui->label_PSU_dissipation_5->setText(zeroChopper(QString::number(val,'f',1)));
	datalogger->setData("PSU_Power_Dissipation", QString::number(val, 'f',1));
}

/**********************************************************************************************************************************************************************************
 * Show mode settings
 * *******************************************************************************************************************************************************************************/

void GUI_1channel_ODC::show_PWM_settings(bool input)
{
	ui->PWM_freq_label->setVisible(input);
	ui->PWM_dc_label->setVisible(input);
	ui->pushButton_PWM_frequency_1->setVisible(input);
	ui->pushButton_PWM_duty_cycle_1->setVisible(input);
}

void GUI_1channel_ODC::show_DLL_settings(bool input)
{
	ui->DLL_stepFreq_label->setVisible(input);
	ui->DLL_threshold_label->setVisible(input);
	ui->pushButton_DLL_step_frequency_1->setVisible(input);
	ui->pushButton_DLL_threshold_1->setVisible(input);
	ui->label_DLL_frequency_lock_1->setVisible(input);
	ui->label_DLL_frequency_lock_2->setVisible(input);
	ui->label_DLL_frequency_lock_3->setVisible(input);
}

void GUI_1channel_ODC::show_ALL_settings(bool input)
{
	ui->ALL_threshold_label->setVisible(input);
	ui->label_3->setVisible(input);
	ui->pushButton_ALL_threshold_1->setVisible(input);
}

void GUI_1channel_ODC::show_autogain_controls(int intrasys_num, bool enable)
{
	/* enable/disable power controls */
	ui->pushButton_power_dbm_1->setEnabled(enable);
	ui->pwr_dbm_label->setEnabled(enable);
	ui->pushButton_power_watt_1->setEnabled(enable);
	ui->pwr_watt_label->setEnabled(enable);
	ui->pushButton_power_dbm_2->setEnabled(enable);
	ui->DLL_power_dbm_label->setEnabled(enable);
	ui->pushButton_power_watt_2->setEnabled(enable);
	ui->DLL_power_watt_label->setEnabled(enable);

	/* If the autogain controls are invalid, so are the sweep controls. */
	ui->SWP_power_dbm_label->setEnabled(enable);
	ui->SWP_power_watt_label->setEnabled(enable);
	ui->pushButton_SWP_power_dbm_1->setEnabled(enable);
	ui->pushButton_SWP_power_watt_1->setEnabled(enable);

	//DO NOT ADD FEED FORWARD CONTROLS HERE, GIVE IT A SEPARATE FUNCTION SO THAT YOU CAN KEEP CONTROLS BLOCK WHILE PID / ANALOGING
	//Ideally separate functions for GCS and MCS and PWRSGD... analog control with IQ Mod, could benefit from GCS controls only for example...
}

void GUI_1channel_ODC::show_GCS_controls(int intrasys_num, bool enable)
{
	ui->label_VGA_attenuation_1->setVisible(enable);
	ui->pushButton_VGA_attenuation_1->setVisible(enable);
}

void GUI_1channel_ODC::show_MCS_controls(int intrasys_num, bool enable)
{
	ui->label_IQMod_magnitude_1->setVisible(enable);
	ui->pushButton_IQMod_magnitude_1->setVisible(enable);
}

/**********************************************************************************************************************************************************************************
 * VERSION COMPATIBILITY CHECK
 * *******************************************************************************************************************************************************************************/
void GUI_1channel_ODC::firmware_version_check(QString firmware_version)
{
	QStringList ver = firmware_version.split(".");
	int firmware_version_actual[4];
	firmware_version_actual[0] = ver.at(0).toInt();
	firmware_version_actual[1] = ver.at(1).toInt();
	firmware_version_actual[2] = ver.at(2).toInt();
	if (ver.count() > 3)
	{
		firmware_version_actual[3] = ver.at(3).toInt();
	}
	else
	{
		firmware_version_actual[3] = 0;
	}

	if(isVersionCompatible(firmware_version_actual,firmware_version_requirement) == false)
	{
		QString comp_warning =	"The current firmware version is unsupported by this application and may result in unpredictable behaviour. "
								"Please upgrade the firmware of your SGx board to at least the recommended version.\n"
								"\nRecommended firmware: v";
		for (int i = 0; i < sizeof(&firmware_version_requirement); i++)
		{
			comp_warning += QString::number(firmware_version_requirement[i]) + ".";
		}
		comp_warning.chop(1);	//chop the redundant period
		comp_warning += "\nCurrent firmware: " + firmware_version;

		QMessageBox message;
		message.warning(nullptr,	"Software imcompatibility", comp_warning);
		show_notification("Firmware is outdated!\nProceed at own risk.");

		this->~GUI_1channel_ODC();
	}
}

/**********************************************************************************************************************************************************************************
 * NOTIFICATION / WARNING BAR
 * *******************************************************************************************************************************************************************************/
void GUI_1channel_ODC::handler_notification_get(int intrasys_num, QString notification)
{
	show_notification(notification);
}

void GUI_1channel_ODC::show_notification(QString message, int time)
{
	notification_timer->setSingleShot(true);
	connect(notification_timer, SIGNAL(timeout()), this, SLOT(close_notification()));
	ui->label_notification->setText(message);
	ui->label_notification->setVisible(true);
	notification_timer->start(time);
}

void GUI_1channel_ODC::close_notification()
{
	ui->label_notification->setVisible(false);
}

void GUI_1channel_ODC::show_error(QString message)
{
	ui->label_error_message->setText(message);
	ui->label_error_message->setVisible(true);
}

void GUI_1channel_ODC::close_error()
{
	ui->label_error_message->setVisible(false);
}

/**********************************************************************************************************************************************************************************
 * MODBUS
 * *******************************************************************************************************************************************************************************/
void GUI_1channel_ODC::on_pushButton_modbus_OFF_1_clicked()
{
	emit signal_set_modbus_enable(false);
	configure_auto_error_clearing();
}

void GUI_1channel_ODC::on_pushButton_modbus_TCP_slave_1_clicked()
{
	emit signal_set_modbus_enable(true);
	configure_auto_error_clearing();
}

void GUI_1channel_ODC::handler_modbus_connected_enable(bool enable, QString address, int port, QString error)
{
	if (enable)
	{
		ui->pushButton_modbus_TCP_slave_1->setChecked(true);
		ui->settings_plainTextEdit->appendPlainText(QString(">\tModbus Server is listening for connections at:\n\tIP:\t\t%1\n\tPort:\t%2").arg(address).arg(port));
	}
	else
	{
		ui->pushButton_modbus_OFF_1->setChecked(true);
		if (error.isEmpty())
		{
			ui->settings_plainTextEdit->appendPlainText(QString(">\tModbus Server has stopped listening for connections."));
		}
		else
		{
			ui->settings_plainTextEdit->appendPlainText(">\tModbus Error:\n\t" + error);
		}
	}
}

void GUI_1channel_ODC::handler_modbus_mode()
{
	if (config->get_support_modbus_mode() == true)
	{
		if (config->get_modbus_enabled() == true)
		{
			ui->pushButton_modbus_TCP_slave_1->click();
		}
		else
		{
			ui->pushButton_modbus_OFF_1->click();
		}
	}
}


/**********************************************************************************************************************************************************************************
 * EtherCAT
 * *******************************************************************************************************************************************************************************/

void GUI_1channel_ODC::on_pushButton_etherCAT_OFF_1_clicked()
{
	emit signal_set_ethercat_enable(false);
	configure_auto_error_clearing();
}

void GUI_1channel_ODC::on_pushButton_etherCAT_ON_1_clicked()
{
	emit signal_set_ethercat_enable(true);
	configure_auto_error_clearing();
}

void GUI_1channel_ODC::handler_ethercat_mode()
{
	if (config->get_support_ethercat_mode() == true)
	{
		if (config->get_ethercat_enabled() == true)
		{
			ui->pushButton_etherCAT_ON_1->click();
		}
		else
		{
			ui->pushButton_etherCAT_OFF_1->click();
		}
	}
}



/**********************************************************************************************************************************************************************************
 * Logging
 * *******************************************************************************************************************************************************************************/

//
//TODO:
//prevent duplicate inputs; requires some kind of state check of the datalogger...
//
void GUI_1channel_ODC::on_pushButton_data_logging_OFF_1_clicked()
{
	datalogger->enable_datalogging(false);
}

void GUI_1channel_ODC::on_pushButton_data_logging_ON_1_clicked()
{
	datalogger->enable_datalogging(true);
}

void GUI_1channel_ODC::handler_datalogging_enable_get(bool enable)
{
	if (enable == true)
	{
		ui->pushButton_data_logging_ON_1->setChecked(true);
		ui->settings_plainTextEdit->appendPlainText(">\tDatalogging Enabled");
	}
	else
	{
		ui->pushButton_data_logging_OFF_1->setChecked(true);
		ui->settings_plainTextEdit->appendPlainText(">\tDatalogging Disabled");
	}
}

void GUI_1channel_ODC::handler_datalogging(bool enable)
{
	if (enable)
	{
		ui->pushButton_data_logging_ON_1->click();
	}
	else
	{
		ui->pushButton_data_logging_OFF_1->click();
	}
}

void GUI_1channel_ODC::handler_datalogging_storage_sufficient(bool valid)
{
	ui->label_low_storage_1->setVisible(!valid);
}

void GUI_1channel_ODC::handler_datalogging_log_deleted(bool success, QString filename)
{
	ui->settings_plainTextEdit->appendPlainText(">\tInsufficient storage for logging. Deleting older log.");
	if (success)
	{
		ui->settings_plainTextEdit->appendPlainText(">\tDeleted old log: " + filename);
//		qDebug() << "Deleted old log: " << filename;
	}
	else
	{
		ui->settings_plainTextEdit->appendPlainText(">\tCould not delete file: " + filename);
//		qDebug() << "Could not delete file" << filename;
	}
}

//void GUI_1channel_ODC::on_pushButton_data_logging_ON_1_toggled(bool checked)
//{
//	if (datalogger->enable_datalogging(checked) == true)
//	{
//		if (checked)
//			ui->settings_plainTextEdit->appendPlainText(">\tDatalogging Enabled");
//		else
//			ui->settings_plainTextEdit->appendPlainText(">\tDatalogging Disabled");
//	}
//	else
//	{
//		ui->settings_plainTextEdit->appendPlainText(">\tError: unable to start Datalogging");
//	}
//}

